#include <bench/Benchmark.h>
#include <tagcoll/TextFormat.h>
#include <tagcoll/stream/sink.h>
#include <tagcoll/coll/simple.h>
#include <tagcoll/coll/fast.h>
#include <tagcoll/coll/patched.h>
//#include <tagcoll/CardinalityStore.h>
//#include <tagcoll/BasicStringDiskIndex.h>

#include <vector>
#include <iostream>
#include <sstream>
#include <math.h>

namespace bench_collection {

using namespace std;
using namespace tagcoll;

/* Some utility random generator functions */

static inline int rnd(int min, int max)
{
	return min + (int) (max * (rand() / (RAND_MAX + 1.0)));
}

static inline double rnd(double min, double max)
{
	return min + (int) (max * (rand() / (RAND_MAX + 1.0)));
}

class CollectionBench : public Benchmark
{
protected:
	// Vector of available tags.  The tags at the beginning are the most
	// popular ones.
	static vector<string> tags;
	// Vector of available items.  The items at the beginning are the ones with
	// most tags.
	static vector<string> items;
	// Sample collection
	static coll::Simple<string, string> coll;
	// Precomputed tagsets
	static vector< std::set<string> > tagsets;
	// Precomputed itemsets
	static vector< std::set<string> > itemsets;

	// Compute a random number between min and max (inclusive), following a
	// geometric distribution with parameter p
	int rndGeom(int min, int max, double p)
	{
		int res = (int)rint((double)min + ((double)max-(double)min)*(log(1.0 - drand48()) / log(p)));
		if (res > max)
			return max;
		else
			return res;
	}
	
	template<typename OUT>
	void outputROCollection(const OUT& cons);
	template<typename COLL>
	void benchROCollection(const COLL& coll);
	template<typename COLL>
	void benchCollection(COLL& coll);

	std::set<string> makeTagset(double p)
	{
		std::set<string> res;
		size_t ntags = rndGeom(1, 20, p);
		for (size_t j = 0; j < ntags; j++)
		{
			int idx = rndGeom(0, tags.size() - 1, 0.01);
			res.insert(tags[idx]);
		}
		return res;
	}

	std::set<string> makeItemset(double p)
	{
		std::set<string> res;
		size_t nitems = rndGeom(1, 10, p);
		for (size_t j = 0; j < nitems; j++)
		{
			int idx = rndGeom(0, items.size() - 1, 0.001);
			res.insert(items[idx]);
		}
		return res;
	}

	CollectionBench(const std::string& name)
		: Benchmark(name)
	{
		if (tags.empty())
		{
			// Create the tag vocabulary
			for (int i = 0; i < 500; i++)
			{
				stringstream tag;
				tag << "tag" << i;
				tags.push_back(tag.str());
			}

			// Create the package set
			for (int i = 0; i < 10000; i++)
			{
				stringstream pkg;
				pkg << "pkg" << i;
				items.push_back(pkg.str());
			}

			// Create the test collection
			for (size_t i = 0; i < items.size(); i++)
				coll.insert(
						wibble::singleton(items[i]),
						makeTagset((((double)items.size() - (double)i) / (double)items.size())/2)
						);
			
			// Precompute tagsets used for benchmarking
			for (size_t i = 0; i < 100; i++)
				tagsets.push_back(makeTagset(0.02));
			
			// Precompute itemsets used for benchmarking
			for (size_t i = 0; i < 100; i++)
				itemsets.push_back(makeItemset(0.02));
		}
	}
};

vector<string> CollectionBench::tags;
vector<string> CollectionBench::items;
coll::Simple<string, string> CollectionBench::coll;
vector< std::set<string> > CollectionBench::tagsets;
vector< std::set<string> > CollectionBench::itemsets;

template<typename OUT>
void CollectionBench::outputROCollection(const OUT& out)
{
	coll.output(out);
}

template<typename COLL>
void CollectionBench::benchROCollection(const COLL& coll)
{
	{
		Timer t = mktimer("hasTag");
		for (size_t i = 0; i < tags.size(); i++)
			coll.hasTag(tags[i]);
	}{
		Timer t = mktimer("getTags[item]");
		for (size_t i = 0; i < items.size(); i++)
			coll.getTagsOfItem(items[i]);
	}{
		Timer t = mktimer("getTags[items]");
		for (size_t i = 0; i < itemsets.size(); i++)
			coll.getTagsOfItems(itemsets[i]);
	}{
		Timer t = mktimer("getItems[tag]");
		for (size_t i = 0; i < tags.size(); i++)
			coll.getItemsHavingTag(tags[i]);
	}{
		Timer t = mktimer("getItems[tags]");
		for (size_t i = 0; i < tagsets.size(); i++)
			coll.getItemsHavingTags(tagsets[i]);
	}{
		Timer t = mktimer("getTaggedItems");
		coll.getTaggedItems();
	}{
		Timer t = mktimer("getAllTags");
		coll.getAllTags();
	}{
		Timer t = mktimer("getCardinality");
		for (size_t i = 0; i < tags.size(); i++)
			coll.getCardinality(tags[i]);
	}{
		Timer t = mktimer("getCompanionTags");
		for (size_t i = 0; i < tagsets.size(); i++)
			coll.getCompanionTags(tagsets[i]);
	}
	// TODO: getRelatedItems
	{
		Timer t = mktimer("output");
		for (size_t i = 0; i < tagsets.size(); i++)
			coll.output(stream::sink());
	}{
		Timer t = mktimer("outputHavingTags");
		for (size_t i = 0; i < tagsets.size(); i++)
			coll.outputHavingTags(tagsets[i], stream::sink());
	}
}

template<typename COLL>
void CollectionBench::benchCollection(COLL& coll)
{
}

class benchSimple : public CollectionBench
{
protected:
	virtual void main()
	{
		Timer main = mktimer("total");

		Timer t1 = mktimer("instantiating collection");
		coll::Simple<string, string> coll;
		t1.done();

		Timer t2 = mktimer("populating collection");
		outputROCollection(inserter(coll));
		t2.done();

		{
			Timer t = mktimer("read only collection");
			benchROCollection(coll);
		}
	}

public:
	benchSimple() : CollectionBench("Simple") {}
};

#if 0
class benchBasicStringDiskIndex : public CollectionBench
{
protected:
	virtual void main()
	{
		Timer main = mktimer("total");
		{
			BasicStringDiskIndexer indexer;

			Timer t1 = mktimer("indexing collection");
			outputROCollection(indexer);
			t1.done();
			
			Timer t2 = mktimer("writing indexed collection");
			indexer.write("bench-basicstringdiskindex.tmp");
			t2.done();
		}

		Timer t3 = mktimer("instantiating indexed collection");
		BasicStringDiskIndex coll("bench-basicstringdiskindex.tmp");
		t3.done();

		{
			Timer t = mktimer("read only collection");
			benchROCollection(coll);
		}
	}

public:
	benchBasicStringDiskIndex() : CollectionBench("BasicStringDiskIndex") {}
	~benchBasicStringDiskIndex() {
		unlink("bench-basicstringdiskindex.tmp");
	}
};
#endif

#if 0
class benchTDBReadonlyDiskIndex : public CollectionBench
{
protected:
	virtual void main()
	{
		Timer main = mktimer("total");
		TrivialConverter<string, string> conv;
		{
			TDBIndexer<string, string> indexer;

			Timer t1 = mktimer("indexing collection");
			outputROCollection(indexer);
			t1.done();
			
			Timer t2 = mktimer("writing indexed collection");
			indexer.writeIndex(conv, conv,
					"bench-TDBReadonlyDiskIndex1.tmp",
					"bench-TDBReadonlyDiskIndex2.tmp");
			t2.done();
		}

		Timer t3 = mktimer("instantiating indexed collection");
		TDBReadonlyDiskIndex<string, string> coll(
				"bench-TDBReadonlyDiskIndex1.tmp",
				"bench-TDBReadonlyDiskIndex2.tmp",
				conv, conv, conv, conv);
		t3.done();

		{
			Timer t = mktimer("read only collection");
			benchROCollection(coll);
		}
	}

public:
	benchTDBReadonlyDiskIndex() : CollectionBench("TDBReadonlyDiskIndex") {}
	~benchTDBReadonlyDiskIndex() {
		unlink("bench-TDBReadonlyDiskIndex1.tmp");
		unlink("bench-TDBReadonlyDiskIndex2.tmp");
	}
};
#endif

class benchFast : public CollectionBench
{
protected:
	virtual void main()
	{
		Timer main = mktimer("total");
		coll::Fast<string, string> coll;

		Timer t1 = mktimer("indexing collection");
		outputROCollection(inserter(coll));
		t1.done();
			
		{
			Timer t = mktimer("read only collection");
			benchROCollection(coll);
		}
	}

public:
	benchFast() : CollectionBench("Fast") {}
};

class benchPatchedOverhead : public CollectionBench
{
protected:
	virtual void main()
	{
		Timer main = mktimer("total");
		coll::Fast<string, string> subcoll;
		coll::Patched< coll::Fast<string, string> > coll(subcoll);

		Timer t1 = mktimer("indexing collection");
		outputROCollection(inserter(subcoll));
		t1.done();
			
		{
			Timer t = mktimer("read only collection");
			benchROCollection(coll);
		}
	}

public:
	benchPatchedOverhead() : CollectionBench("PatchedOverhead") {}
};

class benchPatched : public CollectionBench
{
protected:
	virtual void main()
	{
		Timer main = mktimer("total");
		coll::Fast<string, string> subcoll;
		coll::Patched< coll::Fast<string, string> > coll(subcoll);

		Timer t1 = mktimer("indexing collection");
		outputROCollection(inserter(coll));
		t1.done();
			
		{
			Timer t = mktimer("read only collection");
			benchROCollection(coll);
		}
	}

public:
	benchPatched() : CollectionBench("Patched") {}
};

#if 0
class benchCardinalityStore : public CollectionBench
{
protected:
	virtual void main()
	{
		Timer main = mktimer("total");
		CardinalityStore<string, string> coll;

		Timer t1 = mktimer("indexing collection");
		outputROCollection(coll);
		t1.done();
			
		{
			Timer t = mktimer("read only collection");
			benchROCollection(coll);
		}
	}

public:
	benchCardinalityStore() : CollectionBench("CardinalityStore") {}
};
#endif

class top : public Benchmark
{
public:
	top() : Benchmark("collection")
	{
		addChild(new benchSimple());
		//addChild(new benchBasicStringDiskIndex());
		//addChild(new benchTDBReadonlyDiskIndex());
		addChild(new benchFast());
		addChild(new benchPatchedOverhead());
		//addChild(new benchPatched());
		//addChild(new benchCardinalityStore());
	}
};

static RegisterRoot r(new top());

}

#include <tagcoll/coll/simple.tcc>
#include <tagcoll/coll/fast.tcc>
#include <tagcoll/coll/patched.tcc>

/* vim:set ts=4 sw=4: */
