//
// Copyright(C) 2025 Simon Howard
//
// You can redistribute and/or modify this program under the terms of the
// GNU General Public License version 2 as published by the Free Software
// Foundation, or any later version. This program is distributed WITHOUT
// ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
// FITNESS FOR A PARTICULAR PURPOSE.
//
//
// High score table
//

#include <stdio.h>
#include <string.h>
#include <ctype.h>
#include <assert.h>
#include <errno.h>

#include "sw.h"

#include "hiscore.h"
#include "swgames.h"
#include "swmain.h"
#include "swsound.h"
#include "swtext.h"
#include "swtitle.h"
#include "video.h"

#define HISCORE_FILENAME "hiscores.txt"
#define MAX_HIGH_SCORES 10

#define TABLE_X 9
#define TABLE_Y 1

struct high_score {
	char name[4];
	score_t score;
};

struct high_score high_scores[MAX_HIGH_SCORES] = {
	{"DLC", {6500}},  // David L. Clark
	{"DG",  {6000}},  // Dave Growden (Ox)
	{"JHC", {5500}},  // Jack Cole
	{"JS",  {5000}},  // Jesse Smith
	{"JH",  {4500}},  // Josh Horowitz
	{"CR",  {4000}},  // Christoph Reichenbach
	{"AMJ", {3500}},  // Andrew Jenner
	{"HJM", {3000}},  // Harry Mason
	{"BMB", {2500}},
	{"SDH", {2000}},  // Simon Howard
};

static int UnpackMedals(int input, int *medals, int max_medals)
{
	int i, result = 0;

	for (i = 0; i < max_medals; ++i) {
		if ((input & (1 << i)) != 0) {
			medals[result] = i;
			++result;
		}
	}

	return result;
}

static bool ReadLine(FILE *fs, char *buf, size_t buf_len)
{
	char tmpbuf[8];

	if (feof(fs) || fgets(buf, buf_len, fs) == NULL) {
		return false;
	}
	// Long line? Keep reading chunks until we reach the end
	if (strchr(buf, '\n') == NULL) {
		do {
			fgets(tmpbuf, sizeof(tmpbuf), fs);
		} while (!feof(fs) && strchr(tmpbuf, '\n') == NULL);
	}
	return true;
}

static bool LoadHighScores(const char *filename)
{
	struct high_score *hs;
	char line[64];
	FILE *fs;
	int idx, medals, ribbons;

	fs = fopen(filename, "r");
	if (fs == NULL) {
		return false;
	}

	idx = 0;
	while (idx < MAX_HIGH_SCORES && ReadLine(fs, line, sizeof(line))) {
		hs = &high_scores[idx];
		// Comment?
		if (line[0] == '#') {
			continue;
		}
		if (sscanf(line, "%3s %d %d %d", hs->name,
		           &hs->score.score, &medals, &ribbons) < 4) {
			continue;
		}
		hs->score.medals_nr =
			UnpackMedals(medals, hs->score.medals, 3);
		hs->score.ribbons_nr =
			UnpackMedals(ribbons, hs->score.ribbons, 6);
		++idx;
	}

	fclose(fs);

	return true;
}

static int PackMedals(const int *medals, int cnt)
{
	int i, result = 0;

	for (i = 0; i < cnt; ++i) {
		result |= 1 << medals[i];
	}

	return result;
}

static bool SaveHighScores(const char *filename)
{
	const struct high_score *hs;
	FILE *fs;
	int i;

	fs = fopen(filename, "w");
	if (fs == NULL) {
		perror("Failed to write high scores: fopen");
		return false;
	}
	fprintf(fs, "# Sopwith high scores file, generated by "
	            PACKAGE_STRING"\n");

	for (i = 0; i < MAX_HIGH_SCORES; ++i) {
		hs = &high_scores[i];
		fprintf(fs, "%-3.3s %7d %d %2d\n", hs->name, hs->score.score,
		        PackMedals(hs->score.medals, hs->score.medals_nr),
		        PackMedals(hs->score.ribbons, hs->score.ribbons_nr));
	}

	return true;
}

static const char *HighScoreFilePath(void)
{
	static char *hiscore_file = NULL;
	size_t len;
	char *pref_path;

	if (hiscore_file != NULL) {
		return hiscore_file;
	}

	// If there is a global high scores file on the system, we will
	// use that instead of the home directory one.
	if (strlen(HISCORES_PATH) > 0) {
		FILE *fs;
		char *p = HISCORES_PATH "/" HISCORE_FILENAME;
		fs = fopen(p, "r+");
		if (fs != NULL) {
			fclose(fs);
			hiscore_file = p;
			return hiscore_file;
		}
		// We print a warning message if the file does appear to
		// exist but we can't write to it for some reason.
		if (errno != ENOENT) {
			fprintf(stderr, "Error opening the system-wide high "
			        "scores file (%s): %s. The file might not "
			        "have the necessary permissions set for this "
			        "program to write to it.\n",
			        p, strerror(errno));
		}
	}

	pref_path = Vid_GetPrefPath();
	if (pref_path == NULL) {
		return HISCORE_FILENAME;
	}

	len = strlen(pref_path) + strlen(HISCORE_FILENAME) + 1;
	hiscore_file = calloc(1, len);
	assert(hiscore_file != NULL);
	snprintf(hiscore_file, len, "%s%s", pref_path, HISCORE_FILENAME);
	return hiscore_file;
}

void LoadHighScoreTable(void)
{
	LoadHighScores(HighScoreFilePath());
}

void SaveHighScoreTable(void)
{
	SaveHighScores(HighScoreFilePath());
}

void InitHighScores(void)
{
	score_t *s;
	int i, m, cnt;

	// At startup, every entry in the high score table gets assigned
	// a smattering of random medals and ribbons:
	for (i = 0; i < MAX_HIGH_SCORES; ++i) {
		s = &high_scores[i].score;
		for (m = 0, cnt = 0; m < 2; ++m) {
			if (s->score > rand() % 4500) {
				s->medals[cnt] = m + 1;
				++cnt;
			}
		}
		s->medals_nr = cnt;
		for (m = 0, cnt = 0; m < 6; ++m) {
			if (s->score > rand() % 6500) {
				s->ribbons[cnt] = m;
				++cnt;
			}
		}
		s->ribbons_nr = cnt;
	}

	// Try to load the high score table file, if one exists.
	LoadHighScoreTable();
}

static void DrawHighScore(const struct high_score *hs, int x, int y)
{
	int px, py, m;
	char buf[20];

	swposcur(x, y);
	snprintf(buf, sizeof(buf), "%-3.3s ....... %-4d",
	         hs->name, hs->score.score);
	swputs(buf);

	if (!conf_medals) {
		return;
	}

	px = (x + 16) * 8 + 4;
	py = (SCR_HGHT - 1) - y * 8 + 5;

	for (m = 0; m < hs->score.medals_nr; ++m) {
		symset_t *ss = &symbol_medal[hs->score.medals[m]];
		Vid_DispSymbol(px + m * 8, py, &ss->sym[0],
		               FACTION_PLAYER1);
	}

	// Medal symbols are 12 characters tall, but each line is
	// only 8 characters tall. The top part of the medal symbol
	// gets cut off so that we can pack all the medals in; we do
	// this by drawing a black box over the top of it.
	Vid_Box(px, py, 32, 4, 0);

	for (m = 0; m < hs->score.ribbons_nr; ++m) {
		symset_t *ss = &symbol_ribbon[hs->score.ribbons[m]];
		int rx = (m / 2) * 8, ry = (m % 2) * 4 + 6;
		Vid_DispSymbol(px + 18 + rx, py - ry, &ss->sym[0],
		               FACTION_PLAYER1);
	}
}

void DrawHighScoreTable(void)
{
	int i;

#ifdef HISCORE_DEBUG
	for (i = 0; i < MAX_HIGH_SCORES; ++i) {
		int m;
		high_scores[i].score.medals_nr = 2;
		high_scores[i].score.medals[0] = MEDAL_COMPETENCE;
		high_scores[i].score.medals[1] = MEDAL_VALOUR;
		for (m = 0; m < 6; ++m) {
			high_scores[i].score.ribbons[m] = m;
		}
		high_scores[i].score.ribbons_nr = 6;
	}
#endif

	swcolor(2);
	swposcur(TABLE_X + 3, TABLE_Y);
	swputs("TOP PILOTS");

	swcolor(3);

	// We draw the table bottom up. See comment in DrawHighScore for why.
	for (i = MAX_HIGH_SCORES - 1; i >= 0; i--) {
		DrawHighScore(&high_scores[i], TABLE_X, TABLE_Y + 2 + i);
	}
}

static int CompareHighScores(const struct high_score *a,
                             const struct high_score *b)
{
	if (a->score.score != b->score.score) {
		return a->score.score - b->score.score;
	}
	if (a->score.medals_nr != b->score.medals_nr) {
		return a->score.medals_nr - b->score.medals_nr;
	}
	if (a->score.ribbons_nr != b->score.ribbons_nr) {
		return a->score.ribbons_nr - b->score.ribbons_nr;
	}
	return 0;
}

// NewHighScoreIndex returns the index in the high_scores[] array where the
// new high score should be inserted, or -1 if it is not a new high score.
static int NewHighScoreIndex(const struct high_score *hs)
{
	int i;

	for (i = 0; i < MAX_HIGH_SCORES; ++i) {
		if (CompareHighScores(hs, &high_scores[i]) > 0) {
			return i;
		}
	}

	return -1;
}

static bool IsNewHighScore(const struct high_score *hs)
{
	return NewHighScoreIndex(hs) >= 0;
}

static bool EnterHighScore(struct high_score *hs)
{
	char *p;

	// TODO: Add a new music track for the high score screen?
	soundoff();

	Vid_ClearBuf();
	DrawHighScoreTable();

	swcolor(2);
	swposcur(2, 15);
	swputs("NEW HIGH SCORE!");
	swcolor(3);
	swposcur(2, 17);
	DrawHighScore(hs, TABLE_X, 17);

	swposcur(TABLE_X - 1, 17);
	swputs("[    ]");
	clrprmpt();
	swposcur(TABLE_X, 17);
	swgets(hs->name, sizeof(hs->name) - 1);

	for (p = hs->name; *p != '\0'; ++p) {
		*p = toupper(*p);
	}

	return strlen(hs->name) > 0;
}

// NewHighScore checks if the given score is actually a new high score, and if
// so: prompts the user for their name and adds it.
bool NewHighScore(score_t *s)
{
	struct high_score new_hs = {"", *s};
	int idx;

	// High scores only apply to the default version of the game.
	if (playmode != PLAYMODE_COMPUTER
	 || currgame != &original_level
	 || conf_missiles || conf_wounded || !conf_animals
	 || !conf_big_explosions) {
		return false;
	}
	if (!IsNewHighScore(&new_hs)) {
		return false;
	}
	if (!EnterHighScore(&new_hs)) {
		return false;
	}

	idx = NewHighScoreIndex(&new_hs);
	if (idx < 0) {
		return false;
	}
	memmove(&high_scores[idx + 1], &high_scores[idx],
	        sizeof(struct high_score) * (MAX_HIGH_SCORES - 1 - idx));
	high_scores[idx] = new_hs;
	SaveHighScoreTable();

	return true;
}
