/* -*- Mode: C; indent-tabs-mode: t; c-basic-offset: 4; tab-width: 4 -*- */
/*
 * s51dude - A Downloader/Uploader for 8051 device programmers
 * Copyright (C) 2008 Lucas Chiesa.
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

#include <stdio.h>
#include <stdlib.h>
#include <getopt.h>
#include <string.h>
#ifdef WIN32
#include <lusb0_usb.h>
#else
#include <usb.h>
#endif
#include <libintl.h>
#include <sys/unistd.h>

#include "s51dude.h"
#include "usbtiny.h"
#include "math.h"

#ifdef WIN32
#include <windows.h>
#define sleep(seconds)      Sleep(seconds * 1000) /* from mingw.org list */
#define usleep(useconds)    Sleep(useconds / 1000)
//El de windows:
//Sleep (1000) Espera 1 segundo
//Sleep (10000) Espera 10 segundo

//El de linux:
//sleep (1)  Espera 1 segundo
//sleep (10) Espera 10 segundos
//usleep (1000) Espera 1ms
//usleep (1) Espera 1us
#endif

global_options	options;
int verify_flag = 1;
int verify_chunk_flag = 0;
int verbose_flag = 0;
int dry_run_flag = 0;
int debug_flag = 0;
int eeprom_flag = 0;
int lockbit1_flag = 0;
int lockbit2_flag = 0;
int lockbit3_flag = 0;
unsigned char wlc[] = {0x00, 0x00, 0x00, 0x00};

static unsigned char* code_buffer;
static unsigned char* read_buffer;

usb_dev_handle *usb_handle;
int	sck_period = 10;

void print_if_verbose (char *msg)
{
	if (verbose_flag) fprintf(stderr,"%s\n",msg);
}

inline void print_error(char *msg)
{
	fprintf(stderr,"%s\n",msg);
}

void check_open_file (void)
{
	if (!options.file) {
		print_error (_("ERROR: Can't open file!!"));
		exit_nice();
	}
}

inline void usbtiny_configure(int ind)
{
	usb_control(CONFIGURE_MSG, options.target->val, ind);
}

void enable_programing()
{
	unsigned char res[4];
	usbtiny_trasnparent_spi(options.target->enable_programing,res);
	//usleep(10000);
	sleep(1);
}

void erase_target()
{
	unsigned char res[4];
	usbtiny_trasnparent_spi(options.target->erase,res);
	sleep(1);
}

void program_lockbits()
{
	// Llamamos a la funcion especifica para
	// el target seleccionado.
	(*options.target->write_lockbits) (lockbit1_flag, lockbit2_flag, lockbit3_flag);
}

void read_lockbits()
{
	lockbit1_flag = lockbit2_flag = lockbit3_flag = 0;
	// Algunos micros no tiene lectura de lock bits.
	if (options.target->read_lockbits != NULL) (*options.target->read_lockbits) ();
	else {
		print_error(_("ERROR: You can't read lock bits from the selected target"));
		exit(0);
	}
	if (lockbit1_flag) printf("%s\n",_("LB1 is set"));
	if (lockbit2_flag) printf("%s\n",_("LB2 is set"));
	if (lockbit3_flag) printf("%s\n",_("LB3 is set"));
	if (!lockbit1_flag) printf("%s\n",_("LB1 is not set"));
	if (!lockbit2_flag) printf("%s\n",_("LB2 is not set"));
	if (!lockbit3_flag) printf("%s\n",_("LB3 is not set"));
}

uint16_t read_hex_file()
{

	int	salida_parser;
	IHexRecord buffer_hex;
	uint16_t buffer_size = 0;
	uint16_t max_mem;
	int i;

	// Cambiamos el maximo de memoria dependiendo de la memoria en uso.
	max_mem = (eeprom_flag) ? options.target->mem_eeprom : options.target->mem_micro;


	// Alocamos un buffer del tamano de la memoria del micro.
	code_buffer = (unsigned char*) malloc (max_mem * sizeof(unsigned char));
	if (!code_buffer)  {
		print_error (_("Error: Out of memory."));
		exit_nice();
	}
	// Cargo el buffer con 0xFF.
	memset(code_buffer,0xFF,max_mem);

	i = 1;
	while ( i ) {
		salida_parser = Read_IHexRecord(&buffer_hex, options.file);
		switch (salida_parser) {
			case IHEX_ERROR_FILE:
			case IHEX_ERROR_INVALID_ARGUMENTS:
			case IHEX_ERROR_INVALID_RECORD:
        // This will fail if the last line does not end with \n.
				print_error (_("ERROR: Bad hex file. Make sure it finishes with newline."));
				exit_nice();
				break;
			case IHEX_ERROR_EOF:
				i = 0;
			case IHEX_OK:
				if (buffer_hex.type == IHEX_TYPE_00) {
					// Guardo el ultimo address para saber el tamano del buffer
					// del codigo.
					buffer_size = buffer_hex.address + buffer_hex.dataLen;
					// Valido que el archivo sea menor que la memoria del micro.
					if (buffer_size > max_mem)
					{
						print_error (_("ERROR: file too large for target"));
						exit_nice();
					}
					memcpy(code_buffer+buffer_hex.address, &(buffer_hex.data), buffer_hex.dataLen);
				}
				break;
		}
	}

	if (debug_flag) {
		printf("\n%s\n",_("DEBUG>> Buffer to progam"));
		for (i = 0; i < buffer_size; i++)
			printf("0x%02X, ", code_buffer[i]);
		printf("\n%s\n",_("DEBUG>> END buffer to program"));
	}

	return buffer_size;
}

long round_d(double x) {
  if (x >= 0)
     return (long) (x+0.5);
  return (long) (x-0.5);
}

uint16_t upload(void)
{
	uint16_t buffer_size;
	uint16_t chunk;
	uint16_t last_percentage_update = 0;

	int i,j;
    float completion_perc, increment_perc;
    unsigned char memory_error = 0;

	buffer_size = read_hex_file();

	/* Configure the usbtiny */
	if (!dry_run_flag) {
		if (eeprom_flag == 1)
			usbtiny_configure(options.target->ind_w_eeprom);
		else
			usbtiny_configure(options.target->ind_w_flash);

		enable_programing();
		/*Si target es flash, borrar memoria.*/
		if(eeprom_flag != 1)
            erase_target();
	}
	/* Upload the code to the target device */

    if (!dry_run_flag)
    {
        read_buffer = (unsigned char*) malloc (options.target->max_chunk * sizeof(unsigned char));
        // Seteo la memoria con 0xFF para no usar memoria sin inicializar cuando
        // se usa el dry_run.
        memset(read_buffer,0xFF,options.target->max_chunk);
    }
    completion_perc = 0;
    increment_perc = ( 100.0 * options.target->max_chunk) / buffer_size;
    if (increment_perc > 100)
        increment_perc = 100;
    printf("00%%\n");
	for ( i = 0; i < buffer_size;i+= chunk )
	{
		chunk = options.target->max_chunk;
		if ( chunk > (buffer_size - i) )
			chunk = buffer_size - i;

		if (!dry_run_flag)
		{
			usb_out(USBTINY_FLASH_WRITE,options.target->delay,i,code_buffer + i, chunk, 32 * sck_period + options.target->delay);
			if (verify_chunk_flag)
            {
                usbtiny_configure(options.target->ind_r_flash);
                usb_in(USBTINY_FLASH_READ,0,i,read_buffer, chunk, 32 * sck_period);
                memory_error = memcmp(read_buffer,code_buffer+i,chunk);
                usbtiny_configure(options.target->ind_w_flash);
            }
            if (verbose_flag && !memory_error)
            {
                printf("Upload:\n");
                for (j = 0; j < chunk; j++)
                {
                    printf("0x%02X, ", (code_buffer+i)[j]);
                }

                printf("\nRead:\n");
                for (j = 0; j < chunk; j++)
                {
                    printf("0x%02X, ", (read_buffer)[j]);
                }
                printf("\n");
            }
		}
		completion_perc+=increment_perc;
        if (completion_perc > 100)
            completion_perc = 100;
		if ( ( round_d(completion_perc) / 5) > last_percentage_update )
        {
            last_percentage_update = round_d(completion_perc) / 5;
            printf("%02d%%\n",last_percentage_update*5);
        }
	}
	return buffer_size;
}



void read_mem(unsigned int size_mem)
{
	unsigned int i,chunk;
  //	int16_t max_mem;

	if (debug_flag) printf("%s%i\n","Read mem size: ",size_mem);

	// Cambiamos el maximo de memoria dependiendo de la memoria en uso.
  //	max_mem = (eeprom_flag) ? options.target->mem_eeprom : options.target->mem_micro;

	read_buffer = (unsigned char*) malloc (size_mem * sizeof(unsigned char));
	// Seteo la memoria con 0xFF para no usar memoria sin inicializar cuando
	// se usa el dry_run.
	memset(read_buffer,0xFF,size_mem);

	/* Configure the usbtiny */
	if (!dry_run_flag) {
		if (eeprom_flag == 1)
			usbtiny_configure(options.target->ind_r_eeprom);
		else
			usbtiny_configure(options.target->ind_r_flash);
		enable_programing();
	}

	for ( i = 0; i < size_mem; i+= chunk )
	{
		chunk = options.target->max_chunk;
		if ( chunk > (size_mem - i) )
			chunk = size_mem - i;

		if (!dry_run_flag)
		{
			usb_in(USBTINY_FLASH_READ,0,i,read_buffer + i, chunk, 32 * sck_period);
		}
	}

	if (debug_flag) {
		printf("\n%s\n",_("DEBUG>> Read buffer"));
		for (i = 0; i < size_mem; i++)
			printf("0x%02X, ", read_buffer[i]);
		printf("\n%s\n",_("DEBUG>> END read buffer"));
	}

}

void write_hex_file(uint16_t file_size)
{
	uint16_t i, record_size;

	IHexRecord temp;

	for ( i = 0; i < file_size; i+= record_size )
	{
		record_size = RECORD_SIZE;
		if ( RECORD_SIZE > (file_size - i) )
			record_size = file_size - i;

		New_IHexRecord(0, i, read_buffer+i, record_size, &temp);
		Write_IHexRecord (temp, options.file);
	}

	New_IHexRecord(1, 0, NULL, 0, &temp);
	Write_IHexRecord (temp, options.file);
}


int verify(uint16_t size)
{
	read_mem (size);
	return !(memcmp(read_buffer,code_buffer,size));
}

void print_params()
{
	printf("%s",_("Target part: "));
	printf("%s\n",options.target->text_micro);

	printf("%s",_("Action: "));
	if (options.operation == UPLOAD) printf("%s%s\n",_("UPLOAD: "),options.path);
	if (options.operation == READ) printf("%s%s\n",_("READ: "), options.path);
	if (options.operation == ERASE) printf("%s\n",_("ERASE"));
	if (options.operation == W_LOCKBITS) printf("%s\n",_("WRITE LOCK BITS"));
	if (options.operation == R_LOCKBITS) printf("%s\n",_("READ LOCK BITS"));

	printf("%s",_("Memory: "));
	if (eeprom_flag) printf("%s\n", _("EEPROM"));
	if (!eeprom_flag) printf("%s\n",_("Flash"));


	printf("%s",_("Verify: "));
	if (verify_flag) printf("%s\n", _("YES"));
	if (!verify_flag) printf("%s\n",_("NO"));

	printf("%s",_("verify_chunk_flag: "));
	if (verify_chunk_flag) printf("%s\n", _("YES"));
	if (!verify_chunk_flag) printf("%s\n",_("NO"));

	printf("%s",_("Dry-run: "));
	if (dry_run_flag) printf("%s\n", _("YES"));
	if (!dry_run_flag) printf("%s\n", _("NO"));
}

int main(int argc, char * argv [])
{
	setlocale(LC_ALL, "");
	bindtextdomain("s51dude", LOCALEDIR);
	textdomain("s51dude");

	options.operation = NAA;
	options.target = NULL;

	int aux;

	get_params (argc, argv);
	if (options.target == NULL) {
		print_error (_("Error: You must specify a part name"));
		exit_nice();
	}
	// Valido que el micro tenga EEPROM, si se intenta usar.
	// de no hacerlo te va a dar error de que el archivo no entra en la memoria.
	if (eeprom_flag == 1 && options.target->mem_eeprom == 0) {
		print_error (_("Error: The selected part does not have EEPROM memory"));
		exit_nice();
	}

	if (verbose_flag) print_params();

	if (!dry_run_flag) {
		if ( ! usbtiny_open()) {
			print_error(_("Error: Could not find USBtiny device"));
			exit_nice();
		}
		usbtiny_powerup();
		if (verbose_flag) printf("%s\n",_("* Found USB programmer"));
		usbtiny_send_protocol();
	}

	switch (options.operation) {
		case UPLOAD:
			options.file = fopen (options.path, "r");
			check_open_file();
			aux = upload();
			fclose(options.file);
			if (verify_flag) {
				if (verify(aux)) {
					printf("%s\n",_("* Upload completed successfully!! :)"));
				} else {
					print_error (_("ERROR: Verify error! :("));
				}
			} else {
				printf("%s\n",_("* Upload completed but not verified."));
			}
			break;
		case READ:
			options.file = fopen (options.path, "w");
			check_open_file();
			read_mem((eeprom_flag)?options.target->mem_eeprom:options.target->mem_micro);
			write_hex_file((eeprom_flag)?options.target->mem_eeprom:options.target->mem_micro);
			fclose(options.file);
			printf("%s\n",_("* Memory read."));
			break;
		case ERASE:
			if (!dry_run_flag) {
				usbtiny_configure(options.target->ind_w_flash);
				enable_programing();
				erase_target();
				printf("%s\n",_("* Target erased."));
			}
			break;
		case W_LOCKBITS:
			if (!dry_run_flag) {
				usbtiny_configure(options.target->ind_w_flash);
				enable_programing();
			}
			program_lockbits();
			printf("%s\n",_("* Lock bits programmed."));
			break;
		case R_LOCKBITS:
			if (!dry_run_flag) {
				usbtiny_configure(options.target->ind_w_flash);
				enable_programing();
			}
			read_lockbits();
			break;
		case NAA:
			print_error (_("ERROR: You should give me something to do!"));
			exit_nice();
	}

	if (!dry_run_flag) {
		usbtiny_powerdown ();
		usbtiny_close();
	}
	free_mem();
	return 0;
}

void exit_nice (void)
{
	if (!dry_run_flag && usb_handle) {
		usbtiny_powerdown ();
		usbtiny_close();
	}
	free_mem();
	exit(1);
}

void free_mem (void)
{
	if (read_buffer) free(read_buffer);
	if (code_buffer) free(code_buffer);
	if (usb_handle) free(usb_handle);
	if (options.path) free(options.path);
}
