% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/nnetar.R
\name{nnetar}
\alias{nnetar}
\alias{print.nnetar}
\alias{print.nnetarmodels}
\title{Neural Network Time Series Forecasts}
\usage{
nnetar(
  y,
  p,
  P = 1,
  size,
  repeats = 20,
  xreg = NULL,
  lambda = NULL,
  model = NULL,
  subset = NULL,
  scale.inputs = TRUE,
  x = y,
  ...
)
}
\arguments{
\item{y}{A numeric vector or time series of class \code{ts}.}

\item{p}{Embedding dimension for non-seasonal time series. Number of
non-seasonal lags used as inputs. For non-seasonal time series, the default
is the optimal number of lags (according to the AIC) for a linear AR(p)
model. For seasonal time series, the same method is used but applied to
seasonally adjusted data (from an stl decomposition).}

\item{P}{Number of seasonal lags used as inputs.}

\item{size}{Number of nodes in the hidden layer. Default is half of the
number of input nodes (including external regressors, if given) plus 1.}

\item{repeats}{Number of networks to fit with different random starting
weights. These are then averaged when producing forecasts.}

\item{xreg}{Optionally, a vector or matrix of external regressors, which
must have the same number of rows as \code{y}. Must be numeric.}

\item{lambda}{Box-Cox transformation parameter. If \code{lambda="auto"},
then a transformation is automatically selected using \code{BoxCox.lambda}.
The transformation is ignored if NULL. Otherwise,
data transformed before model is estimated.}

\item{model}{Output from a previous call to \code{nnetar}. If model is
passed, this same model is fitted to \code{y} without re-estimating any
parameters.}

\item{subset}{Optional vector specifying a subset of observations to be used
in the fit. Can be an integer index vector or a logical vector the same
length as \code{y}. All observations are used by default.}

\item{scale.inputs}{If TRUE, inputs are scaled by subtracting the column
means and dividing by their respective standard deviations. If \code{lambda}
is not \code{NULL}, scaling is applied after Box-Cox transformation.}

\item{x}{Deprecated. Included for backwards compatibility.}

\item{\dots}{Other arguments passed to \code{\link[nnet]{nnet}} for
\code{nnetar}.}
}
\value{
Returns an object of class "\code{nnetar}".

The function \code{summary} is used to obtain and print a summary of the
results.

The generic accessor functions \code{fitted.values} and \code{residuals}
extract useful features of the value returned by \code{nnetar}.

\item{model}{A list containing information about the fitted model}
\item{method}{The name of the forecasting method as a character string}
\item{x}{The original time series.}
\item{xreg}{The external regressors used in fitting (if given).}
\item{residuals}{Residuals from the fitted model. That is x minus fitted values.}
\item{fitted}{Fitted values (one-step forecasts)}
\item{...}{Other arguments}
}
\description{
Feed-forward neural networks with a single hidden layer and lagged inputs
for forecasting univariate time series.
}
\details{
A feed-forward neural network is fitted with lagged values of \code{y} as
inputs and a single hidden layer with \code{size} nodes. The inputs are for
lags 1 to \code{p}, and lags \code{m} to \code{mP} where
\code{m=frequency(y)}. If \code{xreg} is provided, its columns are also
used as inputs. If there are missing values in \code{y} or
\code{xreg}, the corresponding rows (and any others which depend on them as
lags) are omitted from the fit. A total of \code{repeats} networks are
fitted, each with random starting weights. These are then averaged when
computing forecasts. The network is trained for one-step forecasting.
Multi-step forecasts are computed recursively.

For non-seasonal data, the fitted model is denoted as an NNAR(p,k) model,
where k is the number of hidden nodes. This is analogous to an AR(p) model
but with nonlinear functions. For seasonal data, the fitted model is called
an NNAR(p,P,k)[m] model, which is analogous to an ARIMA(p,0,0)(P,0,0)[m]
model but with nonlinear functions.
}
\examples{
fit <- nnetar(lynx)
fcast <- forecast(fit)
plot(fcast)

## Arguments can be passed to nnet()
fit <- nnetar(lynx, decay=0.5, maxit=150)
plot(forecast(fit))
lines(lynx)

## Fit model to first 100 years of lynx data
fit <- nnetar(window(lynx,end=1920), decay=0.5, maxit=150)
plot(forecast(fit,h=14))
lines(lynx)

## Apply fitted model to later data, including all optional arguments
fit2 <- nnetar(window(lynx,start=1921), model=fit)

}
\author{
Rob J Hyndman and Gabriel Caceres
}
\keyword{ts}
