/**********************************************************************
 * $read_vecval example -- PLI application using VPI routines
 *
 * C source to read logic values of a net and print the net name and 
 * current logic value.
 *
 * Usage: $read_vecval(<net_name>);
 *
 * For the book, "The Verilog PLI Handbook" by Stuart Sutherland
 *  Book copyright 1999, Kluwer Academic Publishers, Norwell, MA, USA
 *   Contact: www.wkap.il
 *  Example copyright 1998, Sutherland HDL Inc, Portland, Oregon, USA
 *   Contact: www.sutherland.com or (503) 692-0898
 *********************************************************************/

#include <stdlib.h>    /* ANSI C standard library */
#include <stdio.h>     /* ANSI C standard input/output library */
#include "vpi_user.h"  /* IEEE 1364 PLI VPI routine library  */
#include "veriuser.h"  /* IEEE 1364 PLI TF routine library    
                          (using TF routines for simulation control) */

/* prototypes of routines in this PLI application */
int  PLIbook_ReadVecVal_calltf(), PLIbook_ReadVecVal_compiletf();
int  PLIbook_getbit();
char PLIbook_get_4state_val();

/**********************************************************************
 * VPI Registration Data
 *********************************************************************/
void PLIbook_ReadVecVal_register()
{
  s_vpi_systf_data tf_data;
  tf_data.type      = vpiSysTask;
  tf_data.tfname    = "$read_vecval";
  tf_data.calltf    = PLIbook_ReadVecVal_calltf;
  tf_data.compiletf = PLIbook_ReadVecVal_compiletf;
  tf_data.sizetf    = NULL;

  vpi_register_systf(&tf_data);
}
/*********************************************************************/

/**********************************************************************
 * compiletf application
 *********************************************************************/
int PLIbook_ReadVecVal_compiletf(char *user_data)
{
  vpiHandle systf_h, arg_itr, arg_h;
  int       arg_type;
  
  /* obtain a handle to the system task instance */
  systf_h = vpi_handle(vpiSysTfCall, NULL);
  if (systf_h == NULL) {
    vpi_printf("ERROR: $read_vecval failed to obtain systf handle\n");
    tf_dofinish(); /* abort simulation */
    return(0);
  }

  /* obtain handles to system task arguments */
  arg_itr = vpi_iterate(vpiArgument, systf_h);
  if (arg_itr == NULL) {
    vpi_printf("ERROR: $read_vecval requires 1 argument\n");
    tf_dofinish(); /* abort simulation */
    return(0);
  }
  
  /* check the type of object in system task arguments */
  arg_h = vpi_scan(arg_itr);
  arg_type = vpi_get(vpiType, arg_h);
  if (arg_type != vpiNet && arg_type != vpiReg) {
    vpi_printf("ERROR: $read_vecval arg must be a net or reg\n");
    vpi_free_object(arg_itr); /* free iterator memory */
    tf_dofinish(); /* abort simulation */
    return(0);
  }

  /* check that there are no more system task arguments */
  arg_h = vpi_scan(arg_itr);
  if (arg_h != NULL) {
    vpi_printf("ERROR: $read_vecval can only have 1 argument\n");
    vpi_free_object(arg_itr); /* free iterator memory */
    tf_dofinish(); /* abort simulation */
    return(0);
  }
  return(0);
}

/**********************************************************************
 * calltf routine
 *********************************************************************/
int PLIbook_ReadVecVal_calltf(char *user_data) 
{
  vpiHandle   systf_h, arg_itr, arg_h, vector_h;
  s_vpi_value vector_val;     /* structure to receive vector value */
  int i, vector_size, array_size, avalbit, bvalbit, bit_num;
  char vlogval;

  /* obtain a handle to the system task instance */
  systf_h = vpi_handle(vpiSysTfCall, NULL);

  /* obtain handle to system task argument
     compiletf has already verified only 1 arg with correct type */
  arg_itr = vpi_iterate(vpiArgument, systf_h);
  vector_h = vpi_scan(arg_itr);
  vpi_free_object(arg_itr);  /* free iterator memory */

  vector_size = vpi_get(vpiSize, vector_h); /* determine number of...*/
  array_size  = ((vector_size-1) / 32 + 1); /* ...elements in array  */

  vector_val.format = vpiVectorVal;        /* set value format field */

  vpi_get_value(vector_h, &vector_val); /* read vector's logic value */

  vpi_printf("\nVector %s encoded value:\n",
             vpi_get_str(vpiName,vector_h));
  for (i=0; i<array_size; i++) {
    /* the following loop assumes the Verilog LSB is bit 0 */
    for (bit_num=0; bit_num<=31; bit_num++) {
      avalbit=PLIbook_getbit(vector_val.value.vector[i].aval, bit_num);
      bvalbit=PLIbook_getbit(vector_val.value.vector[i].bval, bit_num);
      vlogval=PLIbook_get_4state_val(avalbit, bvalbit);
      vpi_printf("  bit[%2d]  aval/bval = %d/%d  4-state value = %c\n",
                 (i*32+bit_num), avalbit, bvalbit, vlogval);
      /* quit when reach last bit of Verilog vector */
      if ((i*32+bit_num) == vector_size-1) break;
    }
  }
  return(0);
}

/**********************************************************************
 * Function to determine if a specific bit is set in a 32-bit word.
 * Sets the least-significant bit of a mask value to 1 and shifts the
 * mask left to the desired bit number.
 *********************************************************************/
int PLIbook_getbit(int word, int bit_num) 
{
  int mask;
  mask = 0x00000001 << bit_num;
  return((word & mask)? TRUE: FALSE);
}

/**********************************************************************
 * Function to convert aval/bval encoding to 4-state logic represented
 * as a C character.
 *********************************************************************/
char PLIbook_get_4state_val(int aval, int bval)
{
  if      (!bval && !aval) return('0');
  else if (!bval &&  aval) return('1');
  else if ( bval && !aval) return('z');
  else                     return('x');
}

/*********************************************************************/
