-*
   Copyright 2020, Luigi Ferraro, Federico Galetto,
   Francesca Gandini, Hang Huang, Matthew Mastroeni, Xianglong Ni.

   You may redistribute this file under the terms of the GNU General Public
   License as published by the Free Software Foundation, either version 2 of
   the License, or any later version.
*-


document {
	Key => {finiteAction, 
	    (finiteAction, List, PolynomialRing),
	    (finiteAction, Matrix, PolynomialRing),
	    },
	
	Headline => "the group action generated by a list of matrices",
	
	Usage => "finiteAction(L, R), finiteAction(M, R)",
	Inputs => {
	    	"L" => List => {"of matrices representing the generators of a finite group"},
		"M" => Matrix => {"generating a finite cyclic group of matrices"},
		"R" => PolynomialRing => {"on which the group elements 
		    act by linear changes of coordinates"}
		},
	Outputs => {
		FiniteGroupAction => {"the action of the (finite) matrix group generated by 
		    the input matrices on the given polynomial ring"}
		},
	
	"This function is provided by the package ", TO InvariantRing,".",
	
	PARA {
	    "The following example defines the permutation action of
	    a symmetric group on three elements."
	    },
	
	EXAMPLE {
	    	"R = QQ[x_1..x_3]",
		"L = apply(2, i -> permutationMatrix(3, [i + 1, i + 2] ) )",
		"S3 = finiteAction(L, R)",
		},
	
	PARA {
	    "On the other hand, the action below corresponds to a cyclic permutation 
	    of the variables."
	    },
	
	EXAMPLE {
		"P = permutationMatrix toString 231",
		"C3 = finiteAction(P, R)",
		},    
	    }

document {
	Key => {FiniteGroupAction},
	
	Headline => "the class of all finite group actions",
	
	"This class is provided by the package ", TO InvariantRing,".",
	
	PARA {
	    	TT "FiniteGroupAction", " is the class of all finite
		matrix group actions on polynomial rings for the
		purpose of computing invariants.
		It is created using ", TO "finiteAction", ". ",
		"Note that diagonal actions of finite abelian
		groups can be created with the class ",
		TO "DiagonalAction", " which implements more
		efficient methods for computing invariants."
	    },
	}

document {
	Key => {(generators, FiniteGroupAction)},
	Headline => "generators of a finite group",
	Usage => "generators G",
	Inputs => {
	    	"G" => FiniteGroupAction =>
		{"the action of a finite group"},
		},
	Outputs => {
		List => {"a list of generators of the group"}
		},
	"This function is provided by the package ", TO InvariantRing,". ",

    	PARA {
	    "Use this function to get the user-defined
	    generators of a group action."
	},
    
    	PARA { "The following example defines the permutation action
	    of a symmetric group on three elements using three
	    transpositions."  },
	
	EXAMPLE {
	    "R = QQ[x_1..x_3]",
	    "L = {matrix {{0,1,0},{1,0,0},{0,0,1}}, matrix {{0,0,1},{0,1,0},{1,0,0}}, matrix {{1,0,0},{0,0,1},{0,1,0}} }",
	    "G = finiteAction(L, R)",
	    "generators G"
	    },
	    
	    }

document {
	Key => {group, (group, FiniteGroupAction)},
	
	Headline => "list all elements of the group of a finite group action",
	
	Usage => "group G",
	Inputs => {
	    	"G" => FiniteGroupAction
		},
	Outputs => {
		List => {"of all elements in the finite matrix group associated to
		    the given group action"}
		},
	
	"This function is provided by the package ", TO InvariantRing,".",
	
	PARA {
	    "The following example defines the permutation action of
	    a symmetric group on three elements."
	    },
	
	EXAMPLE {
	    	"R = QQ[x_1..x_3]",
		"L = apply(2, i -> permutationMatrix(3, [i + 1, i + 2] ) )",
		"S3 = finiteAction(L, R)",
		"group S3"
		},
	PARA {
	    "The computation of all elements in the group is actually performed by the method ", 
	    TO schreierGraph, " since the process of computing the Schreier graph of the group
	    yields other useful information about the group besides just its elements."
	    },
	
	SeeAlso => {schreierGraph, words}
	    }

document {
	Key => {isAbelian, (isAbelian, FiniteGroupAction)},

	Headline => "check whether a finite matrix group is Abelian",
	
	Usage => "isAbelian G",
	
	Inputs => {
	    	"G" => FiniteGroupAction
		},
	    
	Outputs => {
		Boolean => "whether the group associated to the action is Abelian"
		},
	    
	"This function is provided by the package ", TO InvariantRing,". ",

	PARA {
	    "The following example defines the action of
	    a symmetric group permuting the three variables generating
	    a polynomial ring."
	    },
    	
	EXAMPLE {
	    "R = QQ[x_1..x_4]",
	    "P = apply(3, i -> permutationMatrix(4, [i + 1, i + 2] ) )",
	    "S4 = finiteAction(P, R)",
	    "isAbelian S4",
	    },
	    	    
	PARA {
	    "The following example defines the action of
	    the product of two cyclic groups. One group has order 3
	    and cyclically permutes the three variables generating
	    a polynomial ring. The other group has order 2 and
	    multiplies the variables by -1."
	    },
	
	EXAMPLE {
	    	"R = QQ[x_1..x_3]",
		"L = {matrix {{0,0,1},{0,1,0},{1,0,0}}, matrix {{-1,0,0},{0,-1,0},{0,0,-1}} }",
		"G = finiteAction(L, R)",
		"isAbelian G"
		},
	    }

document {
	Key => {(numgens, FiniteGroupAction)},
	Headline => "number of generators of a finite group",
	Usage => "numgens G",
	Inputs => {
	    	"G" => FiniteGroupAction =>
		{"the action of a finite group"},
		},
	Outputs => {
		ZZ => {"the number of generators of the group"}
		},
	"This function is provided by the package ", TO InvariantRing,". ",

    	PARA {
	    "Use this function to get the number of user-defined
	    generators of a group action."
	},
    
    	PARA { "The following example defines the permutation action
	    of a symmetric group on three elements using three
	    transpositions."  },
	
	EXAMPLE {
	    "R = QQ[x_1..x_3]",
	    "L = {matrix {{0,1,0},{1,0,0},{0,0,1}}, matrix {{0,0,1},{0,1,0},{1,0,0}}, matrix {{1,0,0},{0,0,1},{0,1,0}} }",
	    "G = finiteAction(L, R)",
	    "numgens G"
	    },
	
	Caveat => {"The integer returned by this function is not
	    necessarily the minimal number of generators of the
	    group, rather it is the cardinality of the generating
	    set defined by the user."}
	    
	    }

document {
	Key => {schreierGraph, (schreierGraph, FiniteGroupAction)},
	
	Headline => "Schreier graph of a finite group",
	
	Usage => "schreierGraph G",
	Inputs => {
	    	"G" => FiniteGroupAction => {"a finite group action"},
		},
	Outputs => {
		HashTable => {"representing the Schreier graph of the group"}
		},
	
	"This function is provided by the package ", TO InvariantRing,".",
	
	PARA {
	    "For a finite group action, we form a ", TO "HashTable",
	    " whose keys are the generators provided
	    by the user. The value corresponding to a generator ",
	    TT "g", " is a ", TO "HashTable", " containing all pairs ",
	    TT "a => b", " such that ", TT "a*g == b",
	    ". This represents the Schreier graph of the group
	    relative to the generating set provided by the user."
	    },
	
	PARA {
	    "The following example defines the permutation action of
	    a symmetric group on three elements using only two
	    generators, a transposition and a 3-cycle."
	    },
	
	EXAMPLE {
	    	"R = QQ[x_1..x_3]",
		"L = {matrix {{0,1,0},{1,0,0},{0,0,1}}, matrix {{0,0,1},{0,1,0},{1,0,0}} }",
		"G = finiteAction(L, R)",
		"schreierGraph G"
		},
	    }

document {
	Key => {words, (words, FiniteGroupAction)},
	
	Headline => "associate a word in the generators of a group to each element",
	
	Usage => "words G",
	Inputs => {
	    	"G" => FiniteGroupAction
		},
	Outputs => {
		HashTable => {"associating to each element in the group of the action
		    a word of minimal length in (the indices of) the generators of the group"}
		},
	
	"This function is provided by the package ", TO InvariantRing,".",
	
	PARA {
	    "The following example computes, for each permutation in the symmetric group
	    on three elements, a word of minimal length in the Coxeter generators."
	    },
	
	EXAMPLE {
	    	"R = QQ[x_1..x_3]",
		"L = apply(2, i -> permutationMatrix(3, [i + 1, i + 2] ) )",
		"S3 = finiteAction(L, R)",
		"words S3"
		},
	PARA {
	    "The computation of the words addressing each element in the group is actually 
	    performed by the method ", TO schreierGraph, " since the process of computing 
	    the Schreier graph of the group yields other useful information about the group."
	    },
	
	SeeAlso => {group, schreierGraph}
	    }
	
	
document {
	Key => {permutationMatrix, 
	    (permutationMatrix, String),
	    (permutationMatrix, ZZ, Array),
	    (permutationMatrix, ZZ, List),
	    (permutationMatrix, Array),
	    (permutationMatrix, List)
	    },
	
	Headline => "convert a one-line notation or cyclic notation of a permutation to a matrix representation",
	
	Usage => "permutationMatrix s, permutationMatrix(n , c)",
	Inputs => {
	    	"s" => String =>  {"an array or a list of arrays giving a one-line notation or cyclic notation of a permutation"},
		"n" => ZZ => {"giving the number of intergers getting permuted"},
		"c" => List => {"of arrays giving a cyclic notation of a permutation"}
		},
	Outputs => {
		Matrix => {"the matrix representation of the permutation"}
		},
	
	"This function is provided by the package ", TO InvariantRing,".",
	
	PARA {
	    "The following example converts the one-line notation of a transposition into a matrix representation. "
	    },
	
	EXAMPLE {
		"M = permutationMatrix toString 213",
		},
	
	PARA {
	    "The following example converts the cyclic notation of the same transposition into a matrix representation. Without ",TT "n"," the function assumes ",TT "n"," is the largest integer that appears in your array or list of arrays."
	    },
	
	EXAMPLE {
		"M = permutationMatrix(3,[1,2])",
		"M = permutationMatrix [1,2]",
		},
	PARA {
	    "The following example converts the cyclic notation of a permutation of 4 into a matrix representation."
	    },    
	EXAMPLE {
	    	"M = permutationMatrix(4,{[1,2],[3,4]})",
		"M = permutationMatrix {[1,2],[3,4]}",
	    },
	    }
document {
	Key => {(relations, FiniteGroupAction)},
	Headline => "relations of a finite group",
	Usage => "relations G",
	Inputs => {
	    	"G" => FiniteGroupAction =>
		{"the action of a finite group"},
		},
	Outputs => {
		List => {"a list of relations of the group"}
		},
	"This function is provided by the package ", TO InvariantRing,". ",

    	PARA {
	    "Use this function to get the relations among elements of a group. Each element is represented by a word of minimal length in the Coxter generators. And each relation is represented by a list of two words that equates the group element represented by those two words."
	},
    
    	PARA { "The following example defines the permutation action
	    of a symmetric group on three elements using three
	    transpositions."  },
	
	EXAMPLE {
	    "R = QQ[x_1..x_3]",
	    "L = {matrix {{0,1,0},{1,0,0},{0,0,1}}, matrix {{0,0,1},{0,1,0},{1,0,0}}, matrix {{1,0,0},{0,0,1},{0,1,0}} }",
	    "G = finiteAction(L, R)",
	    "relations G"
	    },
	
	SeeAlso => {group, schreierGraph, words}
	    
	    }
