//===-- lldb-versioning.h ----------------------------------------*- C++
//-*-===//
//
// Part of the LLVM Project, under the Apache License v2.0 with LLVM Exceptions.
// See https://llvm.org/LICENSE.txt for license information.
// SPDX-License-Identifier: Apache-2.0 WITH LLVM-exception
//
//===----------------------------------------------------------------------===//

#ifndef LLDB_LLDB_VERSIONING_H
#define LLDB_LLDB_VERSIONING_H

// LLDB API version
#define LLDB_API_MAJOR_VERSION 1
#define LLDB_API_MINOR_VERSION 0

/*
  API versioning
 ---------------------------------

 The LLDB API is versioned independently of the LLDB source base
 Our API version numbers are composed of a major and a minor number

 The major number means a complete and stable revision of the API. Major numbers
 are compatibility breakers
 (i.e. when we change the API major number, there is no promise of compatibility
 with the previous major version
  and we are free to remove and/or change any APIs)
 Minor numbers are a work-in-progress evolution of the API. APIs will not be
 removed or changed across minor versions
 (minors do not break compatibility). However, we can deprecate APIs in minor
 versions or add new APIs in minor versions
 A deprecated API is supposedly going to be removed in the next major version
 and will generate a warning if used
 APIs we add in minor versions will not be removed (at least until the following
 major) but they might theoretically be deprecated
 in a following minor version
 Users are discouraged from using the LLDB version number to test for API
 features and should instead use the API version checking
 as discussed below

  API version checking
 ---------------------------------

 You can (optionally) sign into an API version checking feature
 To do so you need to define three macros:
 LLDB_API_CHECK_VERSIONING - define to any value (or no value)
 LLDB_API_MAJOR_VERSION_WANTED - which major version of the LLDB API you are
 targeting
 LLDB_API_MINOR_VERSION_WANTED - which minor version of the LLDB API you are
 targeting

 If these macros exist - LLDB will enable version checking of the public API

 If LLDB_API_MAJOR_VERSION is not equal to LLDB_API_MAJOR_VERSION_WANTED we will
 immediately halt your compilation with an error
 This is by design, since we do not make any promise of compatibility across
 major versions - if you really want to test your luck, disable the versioning
 altogether

 If the major version test passes, you have signed up for a specific minor
 version of the API
 Whenever we add or deprecate an API in a minor version, we will mark it with
 either
 LLDB_API_NEW_IN_DOT_x - this API is new in LLDB .x
 LLDB_API_DEPRECATED_IN_DOT_x - this API is deprecated as of .x

 If you are using an API new in DOT_x
  if LLDB_API_MINOR_VERSION_WANTED >= x then all is well, else you will get a
 compilation error
   This is meant to prevent you from using APIs that are newer than whatever
 LLDB you want to target

 If you are using an API deprecated in DOT_x
  if LLDB_API_MINOR_VERSION_WANTED >= x then you will get a compilation warning,
 else all is well
  This is meant to let you know that you are using an API that is deprecated and
 might go away

  Caveats
 ---------------------------------

 Version checking only works on clang on OSX - you will get an error if you try
 to enable it on any other OS/compiler
 If you want to enable version checking on other platforms, you will need to
 define appropriate implementations for
 LLDB_API_IMPL_DEPRECATED and LLDB_API_IMPL_TOONEW and any other infrastructure
 your compiler needs for this purpose

 We have no deprecation-as-error mode

 There is no support for API versioning in Python

 We reserve to use macros whose names begin with LLDB_API_ and you should not
 use them in your source code as they might conflict
 with present or future macro names we are using to implement versioning
*/

// if you want the version checking to work on other OS/compiler, define
// appropriate IMPL_DEPRECATED/IMPL_TOONEW and define
// LLDB_API_CHECK_VERSIONING_WORKS when you are ready to go live
#if defined(__APPLE__) && defined(__clang__)
#define LLDB_API_IMPL_DEPRECATED __attribute__((deprecated))
#define LLDB_API_IMPL_TOONEW __attribute__((unavailable))
#define LLDB_API_CHECK_VERSIONING_WORKS
#endif

#if defined(LLDB_API_CHECK_VERSIONING) &&                                      \
    !defined(LLDB_API_CHECK_VERSIONING_WORKS)
#error                                                                         \
    "API version checking will not work here - please disable or create and submit patches to lldb-versioning.h"
#endif

#if defined(LLDB_API_CHECK_VERSIONING_WORKS) &&                                \
    (!defined(LLDB_API_IMPL_DEPRECATED) || !defined(LLDB_API_IMPL_TOONEW))
#error                                                                         \
    "LLDB_API_CHECK_VERSIONING_WORKS needs LLDB_API_IMPL_DEPRECATED and LLDB_API_IMPL_TOONEW to be defined"
#endif

#if defined(LLDB_API_CHECK_VERSIONING) &&                                      \
    defined(LLDB_API_MAJOR_VERSION_WANTED) &&                                  \
    defined(LLDB_API_MINOR_VERSION_WANTED)

#if defined(LLDB_API_MAJOR_VERSION) &&                                         \
    (LLDB_API_MAJOR_VERSION != LLDB_API_MAJOR_VERSION_WANTED)
#error                                                                         \
    "Cannot link using this LLDB version - public API versions are incompatible"
#endif

#define LLDB_API_MINOR_VERSION_DOT_0 0
#define LLDB_API_MINOR_VERSION_DOT_1 1
#define LLDB_API_MINOR_VERSION_DOT_2 2
#define LLDB_API_MINOR_VERSION_DOT_3 3
#define LLDB_API_MINOR_VERSION_DOT_4 4
#define LLDB_API_MINOR_VERSION_DOT_5 5
#define LLDB_API_MINOR_VERSION_DOT_6 6
#define LLDB_API_MINOR_VERSION_DOT_7 7
#define LLDB_API_MINOR_VERSION_DOT_8 8
#define LLDB_API_MINOR_VERSION_DOT_9 9
#define LLDB_API_MINOR_VERSION_DOT_10 10
#define LLDB_API_MINOR_VERSION_DOT_11 11
#define LLDB_API_MINOR_VERSION_DOT_12 12
#define LLDB_API_MINOR_VERSION_DOT_13 13
#define LLDB_API_MINOR_VERSION_DOT_14 14
#define LLDB_API_MINOR_VERSION_DOT_15 15
#define LLDB_API_MINOR_VERSION_DOT_16 16
#define LLDB_API_MINOR_VERSION_DOT_17 17
#define LLDB_API_MINOR_VERSION_DOT_18 18
#define LLDB_API_MINOR_VERSION_DOT_19 19
#define LLDB_API_MINOR_VERSION_DOT_20 20
#define LLDB_API_MINOR_VERSION_DOT_21 21
#define LLDB_API_MINOR_VERSION_DOT_22 22
#define LLDB_API_MINOR_VERSION_DOT_23 23
#define LLDB_API_MINOR_VERSION_DOT_24 24
#define LLDB_API_MINOR_VERSION_DOT_25 25
#define LLDB_API_MINOR_VERSION_DOT_26 26
#define LLDB_API_MINOR_VERSION_DOT_27 27
#define LLDB_API_MINOR_VERSION_DOT_28 28
#define LLDB_API_MINOR_VERSION_DOT_29 29
#define LLDB_API_MINOR_VERSION_DOT_30 30
#define LLDB_API_MINOR_VERSION_DOT_31 31
#define LLDB_API_MINOR_VERSION_DOT_32 32
#define LLDB_API_MINOR_VERSION_DOT_33 33
#define LLDB_API_MINOR_VERSION_DOT_34 34
#define LLDB_API_MINOR_VERSION_DOT_35 35
#define LLDB_API_MINOR_VERSION_DOT_36 36
#define LLDB_API_MINOR_VERSION_DOT_37 37
#define LLDB_API_MINOR_VERSION_DOT_38 38
#define LLDB_API_MINOR_VERSION_DOT_39 39
#define LLDB_API_MINOR_VERSION_DOT_40 40
#define LLDB_API_MINOR_VERSION_DOT_41 41
#define LLDB_API_MINOR_VERSION_DOT_42 42
#define LLDB_API_MINOR_VERSION_DOT_43 43
#define LLDB_API_MINOR_VERSION_DOT_44 44
#define LLDB_API_MINOR_VERSION_DOT_45 45
#define LLDB_API_MINOR_VERSION_DOT_46 46
#define LLDB_API_MINOR_VERSION_DOT_47 47
#define LLDB_API_MINOR_VERSION_DOT_48 48
#define LLDB_API_MINOR_VERSION_DOT_49 49
#define LLDB_API_MINOR_VERSION_DOT_50 50
#define LLDB_API_MINOR_VERSION_DOT_51 51
#define LLDB_API_MINOR_VERSION_DOT_52 52
#define LLDB_API_MINOR_VERSION_DOT_53 53
#define LLDB_API_MINOR_VERSION_DOT_54 54
#define LLDB_API_MINOR_VERSION_DOT_55 55
#define LLDB_API_MINOR_VERSION_DOT_56 56
#define LLDB_API_MINOR_VERSION_DOT_57 57
#define LLDB_API_MINOR_VERSION_DOT_58 58
#define LLDB_API_MINOR_VERSION_DOT_59 59
#define LLDB_API_MINOR_VERSION_DOT_60 60
#define LLDB_API_MINOR_VERSION_DOT_61 61
#define LLDB_API_MINOR_VERSION_DOT_62 62
#define LLDB_API_MINOR_VERSION_DOT_63 63
#define LLDB_API_MINOR_VERSION_DOT_64 64
#define LLDB_API_MINOR_VERSION_DOT_65 65
#define LLDB_API_MINOR_VERSION_DOT_66 66
#define LLDB_API_MINOR_VERSION_DOT_67 67
#define LLDB_API_MINOR_VERSION_DOT_68 68
#define LLDB_API_MINOR_VERSION_DOT_69 69
#define LLDB_API_MINOR_VERSION_DOT_70 70
#define LLDB_API_MINOR_VERSION_DOT_71 71
#define LLDB_API_MINOR_VERSION_DOT_72 72
#define LLDB_API_MINOR_VERSION_DOT_73 73
#define LLDB_API_MINOR_VERSION_DOT_74 74
#define LLDB_API_MINOR_VERSION_DOT_75 75
#define LLDB_API_MINOR_VERSION_DOT_76 76
#define LLDB_API_MINOR_VERSION_DOT_77 77
#define LLDB_API_MINOR_VERSION_DOT_78 78
#define LLDB_API_MINOR_VERSION_DOT_79 79
#define LLDB_API_MINOR_VERSION_DOT_80 80
#define LLDB_API_MINOR_VERSION_DOT_81 81
#define LLDB_API_MINOR_VERSION_DOT_82 82
#define LLDB_API_MINOR_VERSION_DOT_83 83
#define LLDB_API_MINOR_VERSION_DOT_84 84
#define LLDB_API_MINOR_VERSION_DOT_85 85
#define LLDB_API_MINOR_VERSION_DOT_86 86
#define LLDB_API_MINOR_VERSION_DOT_87 87
#define LLDB_API_MINOR_VERSION_DOT_88 88
#define LLDB_API_MINOR_VERSION_DOT_89 89
#define LLDB_API_MINOR_VERSION_DOT_90 90
#define LLDB_API_MINOR_VERSION_DOT_91 91
#define LLDB_API_MINOR_VERSION_DOT_92 92
#define LLDB_API_MINOR_VERSION_DOT_93 93
#define LLDB_API_MINOR_VERSION_DOT_94 94
#define LLDB_API_MINOR_VERSION_DOT_95 95
#define LLDB_API_MINOR_VERSION_DOT_96 96
#define LLDB_API_MINOR_VERSION_DOT_97 97
#define LLDB_API_MINOR_VERSION_DOT_98 98
#define LLDB_API_MINOR_VERSION_DOT_99 99

#if LLDB_API_MINOR_VERSION_WANTED < LLDB_API_MINOR_VERSION_DOT_0
#define LLDB_API_NEW_IN_DOT_0 LLDB_API_IMPL_TOONEW
#else
#define LLDB_API_NEW_IN_DOT_0
#endif

#if LLDB_API_MINOR_VERSION_WANTED >= LLDB_API_MINOR_VERSION_DOT_0
#define LLDB_API_DEPRECATED_IN_DOT_0 LLDB_API_IMPL_DEPRECATED
#else
#define LLDB_API_DEPRECATED_IN_DOT_0
#endif
#if LLDB_API_MINOR_VERSION_WANTED < LLDB_API_MINOR_VERSION_DOT_1
#define LLDB_API_NEW_IN_DOT_1 LLDB_API_IMPL_TOONEW
#else
#define LLDB_API_NEW_IN_DOT_1
#endif

#if LLDB_API_MINOR_VERSION_WANTED >= LLDB_API_MINOR_VERSION_DOT_1
#define LLDB_API_DEPRECATED_IN_DOT_1 LLDB_API_IMPL_DEPRECATED
#else
#define LLDB_API_DEPRECATED_IN_DOT_1
#endif
#if LLDB_API_MINOR_VERSION_WANTED < LLDB_API_MINOR_VERSION_DOT_2
#define LLDB_API_NEW_IN_DOT_2 LLDB_API_IMPL_TOONEW
#else
#define LLDB_API_NEW_IN_DOT_2
#endif

#if LLDB_API_MINOR_VERSION_WANTED >= LLDB_API_MINOR_VERSION_DOT_2
#define LLDB_API_DEPRECATED_IN_DOT_2 LLDB_API_IMPL_DEPRECATED
#else
#define LLDB_API_DEPRECATED_IN_DOT_2
#endif
#if LLDB_API_MINOR_VERSION_WANTED < LLDB_API_MINOR_VERSION_DOT_3
#define LLDB_API_NEW_IN_DOT_3 LLDB_API_IMPL_TOONEW
#else
#define LLDB_API_NEW_IN_DOT_3
#endif

#if LLDB_API_MINOR_VERSION_WANTED >= LLDB_API_MINOR_VERSION_DOT_3
#define LLDB_API_DEPRECATED_IN_DOT_3 LLDB_API_IMPL_DEPRECATED
#else
#define LLDB_API_DEPRECATED_IN_DOT_3
#endif
#if LLDB_API_MINOR_VERSION_WANTED < LLDB_API_MINOR_VERSION_DOT_4
#define LLDB_API_NEW_IN_DOT_4 LLDB_API_IMPL_TOONEW
#else
#define LLDB_API_NEW_IN_DOT_4
#endif

#if LLDB_API_MINOR_VERSION_WANTED >= LLDB_API_MINOR_VERSION_DOT_4
#define LLDB_API_DEPRECATED_IN_DOT_4 LLDB_API_IMPL_DEPRECATED
#else
#define LLDB_API_DEPRECATED_IN_DOT_4
#endif
#if LLDB_API_MINOR_VERSION_WANTED < LLDB_API_MINOR_VERSION_DOT_5
#define LLDB_API_NEW_IN_DOT_5 LLDB_API_IMPL_TOONEW
#else
#define LLDB_API_NEW_IN_DOT_5
#endif

#if LLDB_API_MINOR_VERSION_WANTED >= LLDB_API_MINOR_VERSION_DOT_5
#define LLDB_API_DEPRECATED_IN_DOT_5 LLDB_API_IMPL_DEPRECATED
#else
#define LLDB_API_DEPRECATED_IN_DOT_5
#endif
#if LLDB_API_MINOR_VERSION_WANTED < LLDB_API_MINOR_VERSION_DOT_6
#define LLDB_API_NEW_IN_DOT_6 LLDB_API_IMPL_TOONEW
#else
#define LLDB_API_NEW_IN_DOT_6
#endif

#if LLDB_API_MINOR_VERSION_WANTED >= LLDB_API_MINOR_VERSION_DOT_6
#define LLDB_API_DEPRECATED_IN_DOT_6 LLDB_API_IMPL_DEPRECATED
#else
#define LLDB_API_DEPRECATED_IN_DOT_6
#endif
#if LLDB_API_MINOR_VERSION_WANTED < LLDB_API_MINOR_VERSION_DOT_7
#define LLDB_API_NEW_IN_DOT_7 LLDB_API_IMPL_TOONEW
#else
#define LLDB_API_NEW_IN_DOT_7
#endif

#if LLDB_API_MINOR_VERSION_WANTED >= LLDB_API_MINOR_VERSION_DOT_7
#define LLDB_API_DEPRECATED_IN_DOT_7 LLDB_API_IMPL_DEPRECATED
#else
#define LLDB_API_DEPRECATED_IN_DOT_7
#endif
#if LLDB_API_MINOR_VERSION_WANTED < LLDB_API_MINOR_VERSION_DOT_8
#define LLDB_API_NEW_IN_DOT_8 LLDB_API_IMPL_TOONEW
#else
#define LLDB_API_NEW_IN_DOT_8
#endif

#if LLDB_API_MINOR_VERSION_WANTED >= LLDB_API_MINOR_VERSION_DOT_8
#define LLDB_API_DEPRECATED_IN_DOT_8 LLDB_API_IMPL_DEPRECATED
#else
#define LLDB_API_DEPRECATED_IN_DOT_8
#endif
#if LLDB_API_MINOR_VERSION_WANTED < LLDB_API_MINOR_VERSION_DOT_9
#define LLDB_API_NEW_IN_DOT_9 LLDB_API_IMPL_TOONEW
#else
#define LLDB_API_NEW_IN_DOT_9
#endif

#if LLDB_API_MINOR_VERSION_WANTED >= LLDB_API_MINOR_VERSION_DOT_9
#define LLDB_API_DEPRECATED_IN_DOT_9 LLDB_API_IMPL_DEPRECATED
#else
#define LLDB_API_DEPRECATED_IN_DOT_9
#endif
#if LLDB_API_MINOR_VERSION_WANTED < LLDB_API_MINOR_VERSION_DOT_10
#define LLDB_API_NEW_IN_DOT_10 LLDB_API_IMPL_TOONEW
#else
#define LLDB_API_NEW_IN_DOT_10
#endif

#if LLDB_API_MINOR_VERSION_WANTED >= LLDB_API_MINOR_VERSION_DOT_10
#define LLDB_API_DEPRECATED_IN_DOT_10 LLDB_API_IMPL_DEPRECATED
#else
#define LLDB_API_DEPRECATED_IN_DOT_10
#endif
#if LLDB_API_MINOR_VERSION_WANTED < LLDB_API_MINOR_VERSION_DOT_11
#define LLDB_API_NEW_IN_DOT_11 LLDB_API_IMPL_TOONEW
#else
#define LLDB_API_NEW_IN_DOT_11
#endif

#if LLDB_API_MINOR_VERSION_WANTED >= LLDB_API_MINOR_VERSION_DOT_11
#define LLDB_API_DEPRECATED_IN_DOT_11 LLDB_API_IMPL_DEPRECATED
#else
#define LLDB_API_DEPRECATED_IN_DOT_11
#endif
#if LLDB_API_MINOR_VERSION_WANTED < LLDB_API_MINOR_VERSION_DOT_12
#define LLDB_API_NEW_IN_DOT_12 LLDB_API_IMPL_TOONEW
#else
#define LLDB_API_NEW_IN_DOT_12
#endif

#if LLDB_API_MINOR_VERSION_WANTED >= LLDB_API_MINOR_VERSION_DOT_12
#define LLDB_API_DEPRECATED_IN_DOT_12 LLDB_API_IMPL_DEPRECATED
#else
#define LLDB_API_DEPRECATED_IN_DOT_12
#endif
#if LLDB_API_MINOR_VERSION_WANTED < LLDB_API_MINOR_VERSION_DOT_13
#define LLDB_API_NEW_IN_DOT_13 LLDB_API_IMPL_TOONEW
#else
#define LLDB_API_NEW_IN_DOT_13
#endif

#if LLDB_API_MINOR_VERSION_WANTED >= LLDB_API_MINOR_VERSION_DOT_13
#define LLDB_API_DEPRECATED_IN_DOT_13 LLDB_API_IMPL_DEPRECATED
#else
#define LLDB_API_DEPRECATED_IN_DOT_13
#endif
#if LLDB_API_MINOR_VERSION_WANTED < LLDB_API_MINOR_VERSION_DOT_14
#define LLDB_API_NEW_IN_DOT_14 LLDB_API_IMPL_TOONEW
#else
#define LLDB_API_NEW_IN_DOT_14
#endif

#if LLDB_API_MINOR_VERSION_WANTED >= LLDB_API_MINOR_VERSION_DOT_14
#define LLDB_API_DEPRECATED_IN_DOT_14 LLDB_API_IMPL_DEPRECATED
#else
#define LLDB_API_DEPRECATED_IN_DOT_14
#endif
#if LLDB_API_MINOR_VERSION_WANTED < LLDB_API_MINOR_VERSION_DOT_15
#define LLDB_API_NEW_IN_DOT_15 LLDB_API_IMPL_TOONEW
#else
#define LLDB_API_NEW_IN_DOT_15
#endif

#if LLDB_API_MINOR_VERSION_WANTED >= LLDB_API_MINOR_VERSION_DOT_15
#define LLDB_API_DEPRECATED_IN_DOT_15 LLDB_API_IMPL_DEPRECATED
#else
#define LLDB_API_DEPRECATED_IN_DOT_15
#endif
#if LLDB_API_MINOR_VERSION_WANTED < LLDB_API_MINOR_VERSION_DOT_16
#define LLDB_API_NEW_IN_DOT_16 LLDB_API_IMPL_TOONEW
#else
#define LLDB_API_NEW_IN_DOT_16
#endif

#if LLDB_API_MINOR_VERSION_WANTED >= LLDB_API_MINOR_VERSION_DOT_16
#define LLDB_API_DEPRECATED_IN_DOT_16 LLDB_API_IMPL_DEPRECATED
#else
#define LLDB_API_DEPRECATED_IN_DOT_16
#endif
#if LLDB_API_MINOR_VERSION_WANTED < LLDB_API_MINOR_VERSION_DOT_17
#define LLDB_API_NEW_IN_DOT_17 LLDB_API_IMPL_TOONEW
#else
#define LLDB_API_NEW_IN_DOT_17
#endif

#if LLDB_API_MINOR_VERSION_WANTED >= LLDB_API_MINOR_VERSION_DOT_17
#define LLDB_API_DEPRECATED_IN_DOT_17 LLDB_API_IMPL_DEPRECATED
#else
#define LLDB_API_DEPRECATED_IN_DOT_17
#endif
#if LLDB_API_MINOR_VERSION_WANTED < LLDB_API_MINOR_VERSION_DOT_18
#define LLDB_API_NEW_IN_DOT_18 LLDB_API_IMPL_TOONEW
#else
#define LLDB_API_NEW_IN_DOT_18
#endif

#if LLDB_API_MINOR_VERSION_WANTED >= LLDB_API_MINOR_VERSION_DOT_18
#define LLDB_API_DEPRECATED_IN_DOT_18 LLDB_API_IMPL_DEPRECATED
#else
#define LLDB_API_DEPRECATED_IN_DOT_18
#endif
#if LLDB_API_MINOR_VERSION_WANTED < LLDB_API_MINOR_VERSION_DOT_19
#define LLDB_API_NEW_IN_DOT_19 LLDB_API_IMPL_TOONEW
#else
#define LLDB_API_NEW_IN_DOT_19
#endif

#if LLDB_API_MINOR_VERSION_WANTED >= LLDB_API_MINOR_VERSION_DOT_19
#define LLDB_API_DEPRECATED_IN_DOT_19 LLDB_API_IMPL_DEPRECATED
#else
#define LLDB_API_DEPRECATED_IN_DOT_19
#endif
#if LLDB_API_MINOR_VERSION_WANTED < LLDB_API_MINOR_VERSION_DOT_20
#define LLDB_API_NEW_IN_DOT_20 LLDB_API_IMPL_TOONEW
#else
#define LLDB_API_NEW_IN_DOT_20
#endif

#if LLDB_API_MINOR_VERSION_WANTED >= LLDB_API_MINOR_VERSION_DOT_20
#define LLDB_API_DEPRECATED_IN_DOT_20 LLDB_API_IMPL_DEPRECATED
#else
#define LLDB_API_DEPRECATED_IN_DOT_20
#endif
#if LLDB_API_MINOR_VERSION_WANTED < LLDB_API_MINOR_VERSION_DOT_21
#define LLDB_API_NEW_IN_DOT_21 LLDB_API_IMPL_TOONEW
#else
#define LLDB_API_NEW_IN_DOT_21
#endif

#if LLDB_API_MINOR_VERSION_WANTED >= LLDB_API_MINOR_VERSION_DOT_21
#define LLDB_API_DEPRECATED_IN_DOT_21 LLDB_API_IMPL_DEPRECATED
#else
#define LLDB_API_DEPRECATED_IN_DOT_21
#endif
#if LLDB_API_MINOR_VERSION_WANTED < LLDB_API_MINOR_VERSION_DOT_22
#define LLDB_API_NEW_IN_DOT_22 LLDB_API_IMPL_TOONEW
#else
#define LLDB_API_NEW_IN_DOT_22
#endif

#if LLDB_API_MINOR_VERSION_WANTED >= LLDB_API_MINOR_VERSION_DOT_22
#define LLDB_API_DEPRECATED_IN_DOT_22 LLDB_API_IMPL_DEPRECATED
#else
#define LLDB_API_DEPRECATED_IN_DOT_22
#endif
#if LLDB_API_MINOR_VERSION_WANTED < LLDB_API_MINOR_VERSION_DOT_23
#define LLDB_API_NEW_IN_DOT_23 LLDB_API_IMPL_TOONEW
#else
#define LLDB_API_NEW_IN_DOT_23
#endif

#if LLDB_API_MINOR_VERSION_WANTED >= LLDB_API_MINOR_VERSION_DOT_23
#define LLDB_API_DEPRECATED_IN_DOT_23 LLDB_API_IMPL_DEPRECATED
#else
#define LLDB_API_DEPRECATED_IN_DOT_23
#endif
#if LLDB_API_MINOR_VERSION_WANTED < LLDB_API_MINOR_VERSION_DOT_24
#define LLDB_API_NEW_IN_DOT_24 LLDB_API_IMPL_TOONEW
#else
#define LLDB_API_NEW_IN_DOT_24
#endif

#if LLDB_API_MINOR_VERSION_WANTED >= LLDB_API_MINOR_VERSION_DOT_24
#define LLDB_API_DEPRECATED_IN_DOT_24 LLDB_API_IMPL_DEPRECATED
#else
#define LLDB_API_DEPRECATED_IN_DOT_24
#endif
#if LLDB_API_MINOR_VERSION_WANTED < LLDB_API_MINOR_VERSION_DOT_25
#define LLDB_API_NEW_IN_DOT_25 LLDB_API_IMPL_TOONEW
#else
#define LLDB_API_NEW_IN_DOT_25
#endif

#if LLDB_API_MINOR_VERSION_WANTED >= LLDB_API_MINOR_VERSION_DOT_25
#define LLDB_API_DEPRECATED_IN_DOT_25 LLDB_API_IMPL_DEPRECATED
#else
#define LLDB_API_DEPRECATED_IN_DOT_25
#endif
#if LLDB_API_MINOR_VERSION_WANTED < LLDB_API_MINOR_VERSION_DOT_26
#define LLDB_API_NEW_IN_DOT_26 LLDB_API_IMPL_TOONEW
#else
#define LLDB_API_NEW_IN_DOT_26
#endif

#if LLDB_API_MINOR_VERSION_WANTED >= LLDB_API_MINOR_VERSION_DOT_26
#define LLDB_API_DEPRECATED_IN_DOT_26 LLDB_API_IMPL_DEPRECATED
#else
#define LLDB_API_DEPRECATED_IN_DOT_26
#endif
#if LLDB_API_MINOR_VERSION_WANTED < LLDB_API_MINOR_VERSION_DOT_27
#define LLDB_API_NEW_IN_DOT_27 LLDB_API_IMPL_TOONEW
#else
#define LLDB_API_NEW_IN_DOT_27
#endif

#if LLDB_API_MINOR_VERSION_WANTED >= LLDB_API_MINOR_VERSION_DOT_27
#define LLDB_API_DEPRECATED_IN_DOT_27 LLDB_API_IMPL_DEPRECATED
#else
#define LLDB_API_DEPRECATED_IN_DOT_27
#endif
#if LLDB_API_MINOR_VERSION_WANTED < LLDB_API_MINOR_VERSION_DOT_28
#define LLDB_API_NEW_IN_DOT_28 LLDB_API_IMPL_TOONEW
#else
#define LLDB_API_NEW_IN_DOT_28
#endif

#if LLDB_API_MINOR_VERSION_WANTED >= LLDB_API_MINOR_VERSION_DOT_28
#define LLDB_API_DEPRECATED_IN_DOT_28 LLDB_API_IMPL_DEPRECATED
#else
#define LLDB_API_DEPRECATED_IN_DOT_28
#endif
#if LLDB_API_MINOR_VERSION_WANTED < LLDB_API_MINOR_VERSION_DOT_29
#define LLDB_API_NEW_IN_DOT_29 LLDB_API_IMPL_TOONEW
#else
#define LLDB_API_NEW_IN_DOT_29
#endif

#if LLDB_API_MINOR_VERSION_WANTED >= LLDB_API_MINOR_VERSION_DOT_29
#define LLDB_API_DEPRECATED_IN_DOT_29 LLDB_API_IMPL_DEPRECATED
#else
#define LLDB_API_DEPRECATED_IN_DOT_29
#endif
#if LLDB_API_MINOR_VERSION_WANTED < LLDB_API_MINOR_VERSION_DOT_30
#define LLDB_API_NEW_IN_DOT_30 LLDB_API_IMPL_TOONEW
#else
#define LLDB_API_NEW_IN_DOT_30
#endif

#if LLDB_API_MINOR_VERSION_WANTED >= LLDB_API_MINOR_VERSION_DOT_30
#define LLDB_API_DEPRECATED_IN_DOT_30 LLDB_API_IMPL_DEPRECATED
#else
#define LLDB_API_DEPRECATED_IN_DOT_30
#endif
#if LLDB_API_MINOR_VERSION_WANTED < LLDB_API_MINOR_VERSION_DOT_31
#define LLDB_API_NEW_IN_DOT_31 LLDB_API_IMPL_TOONEW
#else
#define LLDB_API_NEW_IN_DOT_31
#endif

#if LLDB_API_MINOR_VERSION_WANTED >= LLDB_API_MINOR_VERSION_DOT_31
#define LLDB_API_DEPRECATED_IN_DOT_31 LLDB_API_IMPL_DEPRECATED
#else
#define LLDB_API_DEPRECATED_IN_DOT_31
#endif
#if LLDB_API_MINOR_VERSION_WANTED < LLDB_API_MINOR_VERSION_DOT_32
#define LLDB_API_NEW_IN_DOT_32 LLDB_API_IMPL_TOONEW
#else
#define LLDB_API_NEW_IN_DOT_32
#endif

#if LLDB_API_MINOR_VERSION_WANTED >= LLDB_API_MINOR_VERSION_DOT_32
#define LLDB_API_DEPRECATED_IN_DOT_32 LLDB_API_IMPL_DEPRECATED
#else
#define LLDB_API_DEPRECATED_IN_DOT_32
#endif
#if LLDB_API_MINOR_VERSION_WANTED < LLDB_API_MINOR_VERSION_DOT_33
#define LLDB_API_NEW_IN_DOT_33 LLDB_API_IMPL_TOONEW
#else
#define LLDB_API_NEW_IN_DOT_33
#endif

#if LLDB_API_MINOR_VERSION_WANTED >= LLDB_API_MINOR_VERSION_DOT_33
#define LLDB_API_DEPRECATED_IN_DOT_33 LLDB_API_IMPL_DEPRECATED
#else
#define LLDB_API_DEPRECATED_IN_DOT_33
#endif
#if LLDB_API_MINOR_VERSION_WANTED < LLDB_API_MINOR_VERSION_DOT_34
#define LLDB_API_NEW_IN_DOT_34 LLDB_API_IMPL_TOONEW
#else
#define LLDB_API_NEW_IN_DOT_34
#endif

#if LLDB_API_MINOR_VERSION_WANTED >= LLDB_API_MINOR_VERSION_DOT_34
#define LLDB_API_DEPRECATED_IN_DOT_34 LLDB_API_IMPL_DEPRECATED
#else
#define LLDB_API_DEPRECATED_IN_DOT_34
#endif
#if LLDB_API_MINOR_VERSION_WANTED < LLDB_API_MINOR_VERSION_DOT_35
#define LLDB_API_NEW_IN_DOT_35 LLDB_API_IMPL_TOONEW
#else
#define LLDB_API_NEW_IN_DOT_35
#endif

#if LLDB_API_MINOR_VERSION_WANTED >= LLDB_API_MINOR_VERSION_DOT_35
#define LLDB_API_DEPRECATED_IN_DOT_35 LLDB_API_IMPL_DEPRECATED
#else
#define LLDB_API_DEPRECATED_IN_DOT_35
#endif
#if LLDB_API_MINOR_VERSION_WANTED < LLDB_API_MINOR_VERSION_DOT_36
#define LLDB_API_NEW_IN_DOT_36 LLDB_API_IMPL_TOONEW
#else
#define LLDB_API_NEW_IN_DOT_36
#endif

#if LLDB_API_MINOR_VERSION_WANTED >= LLDB_API_MINOR_VERSION_DOT_36
#define LLDB_API_DEPRECATED_IN_DOT_36 LLDB_API_IMPL_DEPRECATED
#else
#define LLDB_API_DEPRECATED_IN_DOT_36
#endif
#if LLDB_API_MINOR_VERSION_WANTED < LLDB_API_MINOR_VERSION_DOT_37
#define LLDB_API_NEW_IN_DOT_37 LLDB_API_IMPL_TOONEW
#else
#define LLDB_API_NEW_IN_DOT_37
#endif

#if LLDB_API_MINOR_VERSION_WANTED >= LLDB_API_MINOR_VERSION_DOT_37
#define LLDB_API_DEPRECATED_IN_DOT_37 LLDB_API_IMPL_DEPRECATED
#else
#define LLDB_API_DEPRECATED_IN_DOT_37
#endif
#if LLDB_API_MINOR_VERSION_WANTED < LLDB_API_MINOR_VERSION_DOT_38
#define LLDB_API_NEW_IN_DOT_38 LLDB_API_IMPL_TOONEW
#else
#define LLDB_API_NEW_IN_DOT_38
#endif

#if LLDB_API_MINOR_VERSION_WANTED >= LLDB_API_MINOR_VERSION_DOT_38
#define LLDB_API_DEPRECATED_IN_DOT_38 LLDB_API_IMPL_DEPRECATED
#else
#define LLDB_API_DEPRECATED_IN_DOT_38
#endif
#if LLDB_API_MINOR_VERSION_WANTED < LLDB_API_MINOR_VERSION_DOT_39
#define LLDB_API_NEW_IN_DOT_39 LLDB_API_IMPL_TOONEW
#else
#define LLDB_API_NEW_IN_DOT_39
#endif

#if LLDB_API_MINOR_VERSION_WANTED >= LLDB_API_MINOR_VERSION_DOT_39
#define LLDB_API_DEPRECATED_IN_DOT_39 LLDB_API_IMPL_DEPRECATED
#else
#define LLDB_API_DEPRECATED_IN_DOT_39
#endif
#if LLDB_API_MINOR_VERSION_WANTED < LLDB_API_MINOR_VERSION_DOT_40
#define LLDB_API_NEW_IN_DOT_40 LLDB_API_IMPL_TOONEW
#else
#define LLDB_API_NEW_IN_DOT_40
#endif

#if LLDB_API_MINOR_VERSION_WANTED >= LLDB_API_MINOR_VERSION_DOT_40
#define LLDB_API_DEPRECATED_IN_DOT_40 LLDB_API_IMPL_DEPRECATED
#else
#define LLDB_API_DEPRECATED_IN_DOT_40
#endif
#if LLDB_API_MINOR_VERSION_WANTED < LLDB_API_MINOR_VERSION_DOT_41
#define LLDB_API_NEW_IN_DOT_41 LLDB_API_IMPL_TOONEW
#else
#define LLDB_API_NEW_IN_DOT_41
#endif

#if LLDB_API_MINOR_VERSION_WANTED >= LLDB_API_MINOR_VERSION_DOT_41
#define LLDB_API_DEPRECATED_IN_DOT_41 LLDB_API_IMPL_DEPRECATED
#else
#define LLDB_API_DEPRECATED_IN_DOT_41
#endif
#if LLDB_API_MINOR_VERSION_WANTED < LLDB_API_MINOR_VERSION_DOT_42
#define LLDB_API_NEW_IN_DOT_42 LLDB_API_IMPL_TOONEW
#else
#define LLDB_API_NEW_IN_DOT_42
#endif

#if LLDB_API_MINOR_VERSION_WANTED >= LLDB_API_MINOR_VERSION_DOT_42
#define LLDB_API_DEPRECATED_IN_DOT_42 LLDB_API_IMPL_DEPRECATED
#else
#define LLDB_API_DEPRECATED_IN_DOT_42
#endif
#if LLDB_API_MINOR_VERSION_WANTED < LLDB_API_MINOR_VERSION_DOT_43
#define LLDB_API_NEW_IN_DOT_43 LLDB_API_IMPL_TOONEW
#else
#define LLDB_API_NEW_IN_DOT_43
#endif

#if LLDB_API_MINOR_VERSION_WANTED >= LLDB_API_MINOR_VERSION_DOT_43
#define LLDB_API_DEPRECATED_IN_DOT_43 LLDB_API_IMPL_DEPRECATED
#else
#define LLDB_API_DEPRECATED_IN_DOT_43
#endif
#if LLDB_API_MINOR_VERSION_WANTED < LLDB_API_MINOR_VERSION_DOT_44
#define LLDB_API_NEW_IN_DOT_44 LLDB_API_IMPL_TOONEW
#else
#define LLDB_API_NEW_IN_DOT_44
#endif

#if LLDB_API_MINOR_VERSION_WANTED >= LLDB_API_MINOR_VERSION_DOT_44
#define LLDB_API_DEPRECATED_IN_DOT_44 LLDB_API_IMPL_DEPRECATED
#else
#define LLDB_API_DEPRECATED_IN_DOT_44
#endif
#if LLDB_API_MINOR_VERSION_WANTED < LLDB_API_MINOR_VERSION_DOT_45
#define LLDB_API_NEW_IN_DOT_45 LLDB_API_IMPL_TOONEW
#else
#define LLDB_API_NEW_IN_DOT_45
#endif

#if LLDB_API_MINOR_VERSION_WANTED >= LLDB_API_MINOR_VERSION_DOT_45
#define LLDB_API_DEPRECATED_IN_DOT_45 LLDB_API_IMPL_DEPRECATED
#else
#define LLDB_API_DEPRECATED_IN_DOT_45
#endif
#if LLDB_API_MINOR_VERSION_WANTED < LLDB_API_MINOR_VERSION_DOT_46
#define LLDB_API_NEW_IN_DOT_46 LLDB_API_IMPL_TOONEW
#else
#define LLDB_API_NEW_IN_DOT_46
#endif

#if LLDB_API_MINOR_VERSION_WANTED >= LLDB_API_MINOR_VERSION_DOT_46
#define LLDB_API_DEPRECATED_IN_DOT_46 LLDB_API_IMPL_DEPRECATED
#else
#define LLDB_API_DEPRECATED_IN_DOT_46
#endif
#if LLDB_API_MINOR_VERSION_WANTED < LLDB_API_MINOR_VERSION_DOT_47
#define LLDB_API_NEW_IN_DOT_47 LLDB_API_IMPL_TOONEW
#else
#define LLDB_API_NEW_IN_DOT_47
#endif

#if LLDB_API_MINOR_VERSION_WANTED >= LLDB_API_MINOR_VERSION_DOT_47
#define LLDB_API_DEPRECATED_IN_DOT_47 LLDB_API_IMPL_DEPRECATED
#else
#define LLDB_API_DEPRECATED_IN_DOT_47
#endif
#if LLDB_API_MINOR_VERSION_WANTED < LLDB_API_MINOR_VERSION_DOT_48
#define LLDB_API_NEW_IN_DOT_48 LLDB_API_IMPL_TOONEW
#else
#define LLDB_API_NEW_IN_DOT_48
#endif

#if LLDB_API_MINOR_VERSION_WANTED >= LLDB_API_MINOR_VERSION_DOT_48
#define LLDB_API_DEPRECATED_IN_DOT_48 LLDB_API_IMPL_DEPRECATED
#else
#define LLDB_API_DEPRECATED_IN_DOT_48
#endif
#if LLDB_API_MINOR_VERSION_WANTED < LLDB_API_MINOR_VERSION_DOT_49
#define LLDB_API_NEW_IN_DOT_49 LLDB_API_IMPL_TOONEW
#else
#define LLDB_API_NEW_IN_DOT_49
#endif

#if LLDB_API_MINOR_VERSION_WANTED >= LLDB_API_MINOR_VERSION_DOT_49
#define LLDB_API_DEPRECATED_IN_DOT_49 LLDB_API_IMPL_DEPRECATED
#else
#define LLDB_API_DEPRECATED_IN_DOT_49
#endif
#if LLDB_API_MINOR_VERSION_WANTED < LLDB_API_MINOR_VERSION_DOT_50
#define LLDB_API_NEW_IN_DOT_50 LLDB_API_IMPL_TOONEW
#else
#define LLDB_API_NEW_IN_DOT_50
#endif

#if LLDB_API_MINOR_VERSION_WANTED >= LLDB_API_MINOR_VERSION_DOT_50
#define LLDB_API_DEPRECATED_IN_DOT_50 LLDB_API_IMPL_DEPRECATED
#else
#define LLDB_API_DEPRECATED_IN_DOT_50
#endif
#if LLDB_API_MINOR_VERSION_WANTED < LLDB_API_MINOR_VERSION_DOT_51
#define LLDB_API_NEW_IN_DOT_51 LLDB_API_IMPL_TOONEW
#else
#define LLDB_API_NEW_IN_DOT_51
#endif

#if LLDB_API_MINOR_VERSION_WANTED >= LLDB_API_MINOR_VERSION_DOT_51
#define LLDB_API_DEPRECATED_IN_DOT_51 LLDB_API_IMPL_DEPRECATED
#else
#define LLDB_API_DEPRECATED_IN_DOT_51
#endif
#if LLDB_API_MINOR_VERSION_WANTED < LLDB_API_MINOR_VERSION_DOT_52
#define LLDB_API_NEW_IN_DOT_52 LLDB_API_IMPL_TOONEW
#else
#define LLDB_API_NEW_IN_DOT_52
#endif

#if LLDB_API_MINOR_VERSION_WANTED >= LLDB_API_MINOR_VERSION_DOT_52
#define LLDB_API_DEPRECATED_IN_DOT_52 LLDB_API_IMPL_DEPRECATED
#else
#define LLDB_API_DEPRECATED_IN_DOT_52
#endif
#if LLDB_API_MINOR_VERSION_WANTED < LLDB_API_MINOR_VERSION_DOT_53
#define LLDB_API_NEW_IN_DOT_53 LLDB_API_IMPL_TOONEW
#else
#define LLDB_API_NEW_IN_DOT_53
#endif

#if LLDB_API_MINOR_VERSION_WANTED >= LLDB_API_MINOR_VERSION_DOT_53
#define LLDB_API_DEPRECATED_IN_DOT_53 LLDB_API_IMPL_DEPRECATED
#else
#define LLDB_API_DEPRECATED_IN_DOT_53
#endif
#if LLDB_API_MINOR_VERSION_WANTED < LLDB_API_MINOR_VERSION_DOT_54
#define LLDB_API_NEW_IN_DOT_54 LLDB_API_IMPL_TOONEW
#else
#define LLDB_API_NEW_IN_DOT_54
#endif

#if LLDB_API_MINOR_VERSION_WANTED >= LLDB_API_MINOR_VERSION_DOT_54
#define LLDB_API_DEPRECATED_IN_DOT_54 LLDB_API_IMPL_DEPRECATED
#else
#define LLDB_API_DEPRECATED_IN_DOT_54
#endif
#if LLDB_API_MINOR_VERSION_WANTED < LLDB_API_MINOR_VERSION_DOT_55
#define LLDB_API_NEW_IN_DOT_55 LLDB_API_IMPL_TOONEW
#else
#define LLDB_API_NEW_IN_DOT_55
#endif

#if LLDB_API_MINOR_VERSION_WANTED >= LLDB_API_MINOR_VERSION_DOT_55
#define LLDB_API_DEPRECATED_IN_DOT_55 LLDB_API_IMPL_DEPRECATED
#else
#define LLDB_API_DEPRECATED_IN_DOT_55
#endif
#if LLDB_API_MINOR_VERSION_WANTED < LLDB_API_MINOR_VERSION_DOT_56
#define LLDB_API_NEW_IN_DOT_56 LLDB_API_IMPL_TOONEW
#else
#define LLDB_API_NEW_IN_DOT_56
#endif

#if LLDB_API_MINOR_VERSION_WANTED >= LLDB_API_MINOR_VERSION_DOT_56
#define LLDB_API_DEPRECATED_IN_DOT_56 LLDB_API_IMPL_DEPRECATED
#else
#define LLDB_API_DEPRECATED_IN_DOT_56
#endif
#if LLDB_API_MINOR_VERSION_WANTED < LLDB_API_MINOR_VERSION_DOT_57
#define LLDB_API_NEW_IN_DOT_57 LLDB_API_IMPL_TOONEW
#else
#define LLDB_API_NEW_IN_DOT_57
#endif

#if LLDB_API_MINOR_VERSION_WANTED >= LLDB_API_MINOR_VERSION_DOT_57
#define LLDB_API_DEPRECATED_IN_DOT_57 LLDB_API_IMPL_DEPRECATED
#else
#define LLDB_API_DEPRECATED_IN_DOT_57
#endif
#if LLDB_API_MINOR_VERSION_WANTED < LLDB_API_MINOR_VERSION_DOT_58
#define LLDB_API_NEW_IN_DOT_58 LLDB_API_IMPL_TOONEW
#else
#define LLDB_API_NEW_IN_DOT_58
#endif

#if LLDB_API_MINOR_VERSION_WANTED >= LLDB_API_MINOR_VERSION_DOT_58
#define LLDB_API_DEPRECATED_IN_DOT_58 LLDB_API_IMPL_DEPRECATED
#else
#define LLDB_API_DEPRECATED_IN_DOT_58
#endif
#if LLDB_API_MINOR_VERSION_WANTED < LLDB_API_MINOR_VERSION_DOT_59
#define LLDB_API_NEW_IN_DOT_59 LLDB_API_IMPL_TOONEW
#else
#define LLDB_API_NEW_IN_DOT_59
#endif

#if LLDB_API_MINOR_VERSION_WANTED >= LLDB_API_MINOR_VERSION_DOT_59
#define LLDB_API_DEPRECATED_IN_DOT_59 LLDB_API_IMPL_DEPRECATED
#else
#define LLDB_API_DEPRECATED_IN_DOT_59
#endif
#if LLDB_API_MINOR_VERSION_WANTED < LLDB_API_MINOR_VERSION_DOT_60
#define LLDB_API_NEW_IN_DOT_60 LLDB_API_IMPL_TOONEW
#else
#define LLDB_API_NEW_IN_DOT_60
#endif

#if LLDB_API_MINOR_VERSION_WANTED >= LLDB_API_MINOR_VERSION_DOT_60
#define LLDB_API_DEPRECATED_IN_DOT_60 LLDB_API_IMPL_DEPRECATED
#else
#define LLDB_API_DEPRECATED_IN_DOT_60
#endif
#if LLDB_API_MINOR_VERSION_WANTED < LLDB_API_MINOR_VERSION_DOT_61
#define LLDB_API_NEW_IN_DOT_61 LLDB_API_IMPL_TOONEW
#else
#define LLDB_API_NEW_IN_DOT_61
#endif

#if LLDB_API_MINOR_VERSION_WANTED >= LLDB_API_MINOR_VERSION_DOT_61
#define LLDB_API_DEPRECATED_IN_DOT_61 LLDB_API_IMPL_DEPRECATED
#else
#define LLDB_API_DEPRECATED_IN_DOT_61
#endif
#if LLDB_API_MINOR_VERSION_WANTED < LLDB_API_MINOR_VERSION_DOT_62
#define LLDB_API_NEW_IN_DOT_62 LLDB_API_IMPL_TOONEW
#else
#define LLDB_API_NEW_IN_DOT_62
#endif

#if LLDB_API_MINOR_VERSION_WANTED >= LLDB_API_MINOR_VERSION_DOT_62
#define LLDB_API_DEPRECATED_IN_DOT_62 LLDB_API_IMPL_DEPRECATED
#else
#define LLDB_API_DEPRECATED_IN_DOT_62
#endif
#if LLDB_API_MINOR_VERSION_WANTED < LLDB_API_MINOR_VERSION_DOT_63
#define LLDB_API_NEW_IN_DOT_63 LLDB_API_IMPL_TOONEW
#else
#define LLDB_API_NEW_IN_DOT_63
#endif

#if LLDB_API_MINOR_VERSION_WANTED >= LLDB_API_MINOR_VERSION_DOT_63
#define LLDB_API_DEPRECATED_IN_DOT_63 LLDB_API_IMPL_DEPRECATED
#else
#define LLDB_API_DEPRECATED_IN_DOT_63
#endif
#if LLDB_API_MINOR_VERSION_WANTED < LLDB_API_MINOR_VERSION_DOT_64
#define LLDB_API_NEW_IN_DOT_64 LLDB_API_IMPL_TOONEW
#else
#define LLDB_API_NEW_IN_DOT_64
#endif

#if LLDB_API_MINOR_VERSION_WANTED >= LLDB_API_MINOR_VERSION_DOT_64
#define LLDB_API_DEPRECATED_IN_DOT_64 LLDB_API_IMPL_DEPRECATED
#else
#define LLDB_API_DEPRECATED_IN_DOT_64
#endif
#if LLDB_API_MINOR_VERSION_WANTED < LLDB_API_MINOR_VERSION_DOT_65
#define LLDB_API_NEW_IN_DOT_65 LLDB_API_IMPL_TOONEW
#else
#define LLDB_API_NEW_IN_DOT_65
#endif

#if LLDB_API_MINOR_VERSION_WANTED >= LLDB_API_MINOR_VERSION_DOT_65
#define LLDB_API_DEPRECATED_IN_DOT_65 LLDB_API_IMPL_DEPRECATED
#else
#define LLDB_API_DEPRECATED_IN_DOT_65
#endif
#if LLDB_API_MINOR_VERSION_WANTED < LLDB_API_MINOR_VERSION_DOT_66
#define LLDB_API_NEW_IN_DOT_66 LLDB_API_IMPL_TOONEW
#else
#define LLDB_API_NEW_IN_DOT_66
#endif

#if LLDB_API_MINOR_VERSION_WANTED >= LLDB_API_MINOR_VERSION_DOT_66
#define LLDB_API_DEPRECATED_IN_DOT_66 LLDB_API_IMPL_DEPRECATED
#else
#define LLDB_API_DEPRECATED_IN_DOT_66
#endif
#if LLDB_API_MINOR_VERSION_WANTED < LLDB_API_MINOR_VERSION_DOT_67
#define LLDB_API_NEW_IN_DOT_67 LLDB_API_IMPL_TOONEW
#else
#define LLDB_API_NEW_IN_DOT_67
#endif

#if LLDB_API_MINOR_VERSION_WANTED >= LLDB_API_MINOR_VERSION_DOT_67
#define LLDB_API_DEPRECATED_IN_DOT_67 LLDB_API_IMPL_DEPRECATED
#else
#define LLDB_API_DEPRECATED_IN_DOT_67
#endif
#if LLDB_API_MINOR_VERSION_WANTED < LLDB_API_MINOR_VERSION_DOT_68
#define LLDB_API_NEW_IN_DOT_68 LLDB_API_IMPL_TOONEW
#else
#define LLDB_API_NEW_IN_DOT_68
#endif

#if LLDB_API_MINOR_VERSION_WANTED >= LLDB_API_MINOR_VERSION_DOT_68
#define LLDB_API_DEPRECATED_IN_DOT_68 LLDB_API_IMPL_DEPRECATED
#else
#define LLDB_API_DEPRECATED_IN_DOT_68
#endif
#if LLDB_API_MINOR_VERSION_WANTED < LLDB_API_MINOR_VERSION_DOT_69
#define LLDB_API_NEW_IN_DOT_69 LLDB_API_IMPL_TOONEW
#else
#define LLDB_API_NEW_IN_DOT_69
#endif

#if LLDB_API_MINOR_VERSION_WANTED >= LLDB_API_MINOR_VERSION_DOT_69
#define LLDB_API_DEPRECATED_IN_DOT_69 LLDB_API_IMPL_DEPRECATED
#else
#define LLDB_API_DEPRECATED_IN_DOT_69
#endif
#if LLDB_API_MINOR_VERSION_WANTED < LLDB_API_MINOR_VERSION_DOT_70
#define LLDB_API_NEW_IN_DOT_70 LLDB_API_IMPL_TOONEW
#else
#define LLDB_API_NEW_IN_DOT_70
#endif

#if LLDB_API_MINOR_VERSION_WANTED >= LLDB_API_MINOR_VERSION_DOT_70
#define LLDB_API_DEPRECATED_IN_DOT_70 LLDB_API_IMPL_DEPRECATED
#else
#define LLDB_API_DEPRECATED_IN_DOT_70
#endif
#if LLDB_API_MINOR_VERSION_WANTED < LLDB_API_MINOR_VERSION_DOT_71
#define LLDB_API_NEW_IN_DOT_71 LLDB_API_IMPL_TOONEW
#else
#define LLDB_API_NEW_IN_DOT_71
#endif

#if LLDB_API_MINOR_VERSION_WANTED >= LLDB_API_MINOR_VERSION_DOT_71
#define LLDB_API_DEPRECATED_IN_DOT_71 LLDB_API_IMPL_DEPRECATED
#else
#define LLDB_API_DEPRECATED_IN_DOT_71
#endif
#if LLDB_API_MINOR_VERSION_WANTED < LLDB_API_MINOR_VERSION_DOT_72
#define LLDB_API_NEW_IN_DOT_72 LLDB_API_IMPL_TOONEW
#else
#define LLDB_API_NEW_IN_DOT_72
#endif

#if LLDB_API_MINOR_VERSION_WANTED >= LLDB_API_MINOR_VERSION_DOT_72
#define LLDB_API_DEPRECATED_IN_DOT_72 LLDB_API_IMPL_DEPRECATED
#else
#define LLDB_API_DEPRECATED_IN_DOT_72
#endif
#if LLDB_API_MINOR_VERSION_WANTED < LLDB_API_MINOR_VERSION_DOT_73
#define LLDB_API_NEW_IN_DOT_73 LLDB_API_IMPL_TOONEW
#else
#define LLDB_API_NEW_IN_DOT_73
#endif

#if LLDB_API_MINOR_VERSION_WANTED >= LLDB_API_MINOR_VERSION_DOT_73
#define LLDB_API_DEPRECATED_IN_DOT_73 LLDB_API_IMPL_DEPRECATED
#else
#define LLDB_API_DEPRECATED_IN_DOT_73
#endif
#if LLDB_API_MINOR_VERSION_WANTED < LLDB_API_MINOR_VERSION_DOT_74
#define LLDB_API_NEW_IN_DOT_74 LLDB_API_IMPL_TOONEW
#else
#define LLDB_API_NEW_IN_DOT_74
#endif

#if LLDB_API_MINOR_VERSION_WANTED >= LLDB_API_MINOR_VERSION_DOT_74
#define LLDB_API_DEPRECATED_IN_DOT_74 LLDB_API_IMPL_DEPRECATED
#else
#define LLDB_API_DEPRECATED_IN_DOT_74
#endif
#if LLDB_API_MINOR_VERSION_WANTED < LLDB_API_MINOR_VERSION_DOT_75
#define LLDB_API_NEW_IN_DOT_75 LLDB_API_IMPL_TOONEW
#else
#define LLDB_API_NEW_IN_DOT_75
#endif

#if LLDB_API_MINOR_VERSION_WANTED >= LLDB_API_MINOR_VERSION_DOT_75
#define LLDB_API_DEPRECATED_IN_DOT_75 LLDB_API_IMPL_DEPRECATED
#else
#define LLDB_API_DEPRECATED_IN_DOT_75
#endif
#if LLDB_API_MINOR_VERSION_WANTED < LLDB_API_MINOR_VERSION_DOT_76
#define LLDB_API_NEW_IN_DOT_76 LLDB_API_IMPL_TOONEW
#else
#define LLDB_API_NEW_IN_DOT_76
#endif

#if LLDB_API_MINOR_VERSION_WANTED >= LLDB_API_MINOR_VERSION_DOT_76
#define LLDB_API_DEPRECATED_IN_DOT_76 LLDB_API_IMPL_DEPRECATED
#else
#define LLDB_API_DEPRECATED_IN_DOT_76
#endif
#if LLDB_API_MINOR_VERSION_WANTED < LLDB_API_MINOR_VERSION_DOT_77
#define LLDB_API_NEW_IN_DOT_77 LLDB_API_IMPL_TOONEW
#else
#define LLDB_API_NEW_IN_DOT_77
#endif

#if LLDB_API_MINOR_VERSION_WANTED >= LLDB_API_MINOR_VERSION_DOT_77
#define LLDB_API_DEPRECATED_IN_DOT_77 LLDB_API_IMPL_DEPRECATED
#else
#define LLDB_API_DEPRECATED_IN_DOT_77
#endif
#if LLDB_API_MINOR_VERSION_WANTED < LLDB_API_MINOR_VERSION_DOT_78
#define LLDB_API_NEW_IN_DOT_78 LLDB_API_IMPL_TOONEW
#else
#define LLDB_API_NEW_IN_DOT_78
#endif

#if LLDB_API_MINOR_VERSION_WANTED >= LLDB_API_MINOR_VERSION_DOT_78
#define LLDB_API_DEPRECATED_IN_DOT_78 LLDB_API_IMPL_DEPRECATED
#else
#define LLDB_API_DEPRECATED_IN_DOT_78
#endif
#if LLDB_API_MINOR_VERSION_WANTED < LLDB_API_MINOR_VERSION_DOT_79
#define LLDB_API_NEW_IN_DOT_79 LLDB_API_IMPL_TOONEW
#else
#define LLDB_API_NEW_IN_DOT_79
#endif

#if LLDB_API_MINOR_VERSION_WANTED >= LLDB_API_MINOR_VERSION_DOT_79
#define LLDB_API_DEPRECATED_IN_DOT_79 LLDB_API_IMPL_DEPRECATED
#else
#define LLDB_API_DEPRECATED_IN_DOT_79
#endif
#if LLDB_API_MINOR_VERSION_WANTED < LLDB_API_MINOR_VERSION_DOT_80
#define LLDB_API_NEW_IN_DOT_80 LLDB_API_IMPL_TOONEW
#else
#define LLDB_API_NEW_IN_DOT_80
#endif

#if LLDB_API_MINOR_VERSION_WANTED >= LLDB_API_MINOR_VERSION_DOT_80
#define LLDB_API_DEPRECATED_IN_DOT_80 LLDB_API_IMPL_DEPRECATED
#else
#define LLDB_API_DEPRECATED_IN_DOT_80
#endif
#if LLDB_API_MINOR_VERSION_WANTED < LLDB_API_MINOR_VERSION_DOT_81
#define LLDB_API_NEW_IN_DOT_81 LLDB_API_IMPL_TOONEW
#else
#define LLDB_API_NEW_IN_DOT_81
#endif

#if LLDB_API_MINOR_VERSION_WANTED >= LLDB_API_MINOR_VERSION_DOT_81
#define LLDB_API_DEPRECATED_IN_DOT_81 LLDB_API_IMPL_DEPRECATED
#else
#define LLDB_API_DEPRECATED_IN_DOT_81
#endif
#if LLDB_API_MINOR_VERSION_WANTED < LLDB_API_MINOR_VERSION_DOT_82
#define LLDB_API_NEW_IN_DOT_82 LLDB_API_IMPL_TOONEW
#else
#define LLDB_API_NEW_IN_DOT_82
#endif

#if LLDB_API_MINOR_VERSION_WANTED >= LLDB_API_MINOR_VERSION_DOT_82
#define LLDB_API_DEPRECATED_IN_DOT_82 LLDB_API_IMPL_DEPRECATED
#else
#define LLDB_API_DEPRECATED_IN_DOT_82
#endif
#if LLDB_API_MINOR_VERSION_WANTED < LLDB_API_MINOR_VERSION_DOT_83
#define LLDB_API_NEW_IN_DOT_83 LLDB_API_IMPL_TOONEW
#else
#define LLDB_API_NEW_IN_DOT_83
#endif

#if LLDB_API_MINOR_VERSION_WANTED >= LLDB_API_MINOR_VERSION_DOT_83
#define LLDB_API_DEPRECATED_IN_DOT_83 LLDB_API_IMPL_DEPRECATED
#else
#define LLDB_API_DEPRECATED_IN_DOT_83
#endif
#if LLDB_API_MINOR_VERSION_WANTED < LLDB_API_MINOR_VERSION_DOT_84
#define LLDB_API_NEW_IN_DOT_84 LLDB_API_IMPL_TOONEW
#else
#define LLDB_API_NEW_IN_DOT_84
#endif

#if LLDB_API_MINOR_VERSION_WANTED >= LLDB_API_MINOR_VERSION_DOT_84
#define LLDB_API_DEPRECATED_IN_DOT_84 LLDB_API_IMPL_DEPRECATED
#else
#define LLDB_API_DEPRECATED_IN_DOT_84
#endif
#if LLDB_API_MINOR_VERSION_WANTED < LLDB_API_MINOR_VERSION_DOT_85
#define LLDB_API_NEW_IN_DOT_85 LLDB_API_IMPL_TOONEW
#else
#define LLDB_API_NEW_IN_DOT_85
#endif

#if LLDB_API_MINOR_VERSION_WANTED >= LLDB_API_MINOR_VERSION_DOT_85
#define LLDB_API_DEPRECATED_IN_DOT_85 LLDB_API_IMPL_DEPRECATED
#else
#define LLDB_API_DEPRECATED_IN_DOT_85
#endif
#if LLDB_API_MINOR_VERSION_WANTED < LLDB_API_MINOR_VERSION_DOT_86
#define LLDB_API_NEW_IN_DOT_86 LLDB_API_IMPL_TOONEW
#else
#define LLDB_API_NEW_IN_DOT_86
#endif

#if LLDB_API_MINOR_VERSION_WANTED >= LLDB_API_MINOR_VERSION_DOT_86
#define LLDB_API_DEPRECATED_IN_DOT_86 LLDB_API_IMPL_DEPRECATED
#else
#define LLDB_API_DEPRECATED_IN_DOT_86
#endif
#if LLDB_API_MINOR_VERSION_WANTED < LLDB_API_MINOR_VERSION_DOT_87
#define LLDB_API_NEW_IN_DOT_87 LLDB_API_IMPL_TOONEW
#else
#define LLDB_API_NEW_IN_DOT_87
#endif

#if LLDB_API_MINOR_VERSION_WANTED >= LLDB_API_MINOR_VERSION_DOT_87
#define LLDB_API_DEPRECATED_IN_DOT_87 LLDB_API_IMPL_DEPRECATED
#else
#define LLDB_API_DEPRECATED_IN_DOT_87
#endif
#if LLDB_API_MINOR_VERSION_WANTED < LLDB_API_MINOR_VERSION_DOT_88
#define LLDB_API_NEW_IN_DOT_88 LLDB_API_IMPL_TOONEW
#else
#define LLDB_API_NEW_IN_DOT_88
#endif

#if LLDB_API_MINOR_VERSION_WANTED >= LLDB_API_MINOR_VERSION_DOT_88
#define LLDB_API_DEPRECATED_IN_DOT_88 LLDB_API_IMPL_DEPRECATED
#else
#define LLDB_API_DEPRECATED_IN_DOT_88
#endif
#if LLDB_API_MINOR_VERSION_WANTED < LLDB_API_MINOR_VERSION_DOT_89
#define LLDB_API_NEW_IN_DOT_89 LLDB_API_IMPL_TOONEW
#else
#define LLDB_API_NEW_IN_DOT_89
#endif

#if LLDB_API_MINOR_VERSION_WANTED >= LLDB_API_MINOR_VERSION_DOT_89
#define LLDB_API_DEPRECATED_IN_DOT_89 LLDB_API_IMPL_DEPRECATED
#else
#define LLDB_API_DEPRECATED_IN_DOT_89
#endif
#if LLDB_API_MINOR_VERSION_WANTED < LLDB_API_MINOR_VERSION_DOT_90
#define LLDB_API_NEW_IN_DOT_90 LLDB_API_IMPL_TOONEW
#else
#define LLDB_API_NEW_IN_DOT_90
#endif

#if LLDB_API_MINOR_VERSION_WANTED >= LLDB_API_MINOR_VERSION_DOT_90
#define LLDB_API_DEPRECATED_IN_DOT_90 LLDB_API_IMPL_DEPRECATED
#else
#define LLDB_API_DEPRECATED_IN_DOT_90
#endif
#if LLDB_API_MINOR_VERSION_WANTED < LLDB_API_MINOR_VERSION_DOT_91
#define LLDB_API_NEW_IN_DOT_91 LLDB_API_IMPL_TOONEW
#else
#define LLDB_API_NEW_IN_DOT_91
#endif

#if LLDB_API_MINOR_VERSION_WANTED >= LLDB_API_MINOR_VERSION_DOT_91
#define LLDB_API_DEPRECATED_IN_DOT_91 LLDB_API_IMPL_DEPRECATED
#else
#define LLDB_API_DEPRECATED_IN_DOT_91
#endif
#if LLDB_API_MINOR_VERSION_WANTED < LLDB_API_MINOR_VERSION_DOT_92
#define LLDB_API_NEW_IN_DOT_92 LLDB_API_IMPL_TOONEW
#else
#define LLDB_API_NEW_IN_DOT_92
#endif

#if LLDB_API_MINOR_VERSION_WANTED >= LLDB_API_MINOR_VERSION_DOT_92
#define LLDB_API_DEPRECATED_IN_DOT_92 LLDB_API_IMPL_DEPRECATED
#else
#define LLDB_API_DEPRECATED_IN_DOT_92
#endif
#if LLDB_API_MINOR_VERSION_WANTED < LLDB_API_MINOR_VERSION_DOT_93
#define LLDB_API_NEW_IN_DOT_93 LLDB_API_IMPL_TOONEW
#else
#define LLDB_API_NEW_IN_DOT_93
#endif

#if LLDB_API_MINOR_VERSION_WANTED >= LLDB_API_MINOR_VERSION_DOT_93
#define LLDB_API_DEPRECATED_IN_DOT_93 LLDB_API_IMPL_DEPRECATED
#else
#define LLDB_API_DEPRECATED_IN_DOT_93
#endif
#if LLDB_API_MINOR_VERSION_WANTED < LLDB_API_MINOR_VERSION_DOT_94
#define LLDB_API_NEW_IN_DOT_94 LLDB_API_IMPL_TOONEW
#else
#define LLDB_API_NEW_IN_DOT_94
#endif

#if LLDB_API_MINOR_VERSION_WANTED >= LLDB_API_MINOR_VERSION_DOT_94
#define LLDB_API_DEPRECATED_IN_DOT_94 LLDB_API_IMPL_DEPRECATED
#else
#define LLDB_API_DEPRECATED_IN_DOT_94
#endif
#if LLDB_API_MINOR_VERSION_WANTED < LLDB_API_MINOR_VERSION_DOT_95
#define LLDB_API_NEW_IN_DOT_95 LLDB_API_IMPL_TOONEW
#else
#define LLDB_API_NEW_IN_DOT_95
#endif

#if LLDB_API_MINOR_VERSION_WANTED >= LLDB_API_MINOR_VERSION_DOT_95
#define LLDB_API_DEPRECATED_IN_DOT_95 LLDB_API_IMPL_DEPRECATED
#else
#define LLDB_API_DEPRECATED_IN_DOT_95
#endif
#if LLDB_API_MINOR_VERSION_WANTED < LLDB_API_MINOR_VERSION_DOT_96
#define LLDB_API_NEW_IN_DOT_96 LLDB_API_IMPL_TOONEW
#else
#define LLDB_API_NEW_IN_DOT_96
#endif

#if LLDB_API_MINOR_VERSION_WANTED >= LLDB_API_MINOR_VERSION_DOT_96
#define LLDB_API_DEPRECATED_IN_DOT_96 LLDB_API_IMPL_DEPRECATED
#else
#define LLDB_API_DEPRECATED_IN_DOT_96
#endif
#if LLDB_API_MINOR_VERSION_WANTED < LLDB_API_MINOR_VERSION_DOT_97
#define LLDB_API_NEW_IN_DOT_97 LLDB_API_IMPL_TOONEW
#else
#define LLDB_API_NEW_IN_DOT_97
#endif

#if LLDB_API_MINOR_VERSION_WANTED >= LLDB_API_MINOR_VERSION_DOT_97
#define LLDB_API_DEPRECATED_IN_DOT_97 LLDB_API_IMPL_DEPRECATED
#else
#define LLDB_API_DEPRECATED_IN_DOT_97
#endif
#if LLDB_API_MINOR_VERSION_WANTED < LLDB_API_MINOR_VERSION_DOT_98
#define LLDB_API_NEW_IN_DOT_98 LLDB_API_IMPL_TOONEW
#else
#define LLDB_API_NEW_IN_DOT_98
#endif

#if LLDB_API_MINOR_VERSION_WANTED >= LLDB_API_MINOR_VERSION_DOT_98
#define LLDB_API_DEPRECATED_IN_DOT_98 LLDB_API_IMPL_DEPRECATED
#else
#define LLDB_API_DEPRECATED_IN_DOT_98
#endif
#if LLDB_API_MINOR_VERSION_WANTED < LLDB_API_MINOR_VERSION_DOT_99
#define LLDB_API_NEW_IN_DOT_99 LLDB_API_IMPL_TOONEW
#else
#define LLDB_API_NEW_IN_DOT_99
#endif

#if LLDB_API_MINOR_VERSION_WANTED >= LLDB_API_MINOR_VERSION_DOT_99
#define LLDB_API_DEPRECATED_IN_DOT_99 LLDB_API_IMPL_DEPRECATED
#else
#define LLDB_API_DEPRECATED_IN_DOT_99
#endif

#else // defined(LLDB_CHECK_API_VERSIONING) &&
      // defined(LLDB_API_MAJOR_VERSION_WANTED) &&
      // defined(LLDB_API_MINOR_VERSION_WANTED) && defined
      // (LLDB_API_MAJOR_VERSION)

#define LLDB_API_NEW_IN_DOT_0
#define LLDB_API_DEPRECATED_IN_DOT_0
#define LLDB_API_NEW_IN_DOT_1
#define LLDB_API_DEPRECATED_IN_DOT_1
#define LLDB_API_NEW_IN_DOT_2
#define LLDB_API_DEPRECATED_IN_DOT_2
#define LLDB_API_NEW_IN_DOT_3
#define LLDB_API_DEPRECATED_IN_DOT_3
#define LLDB_API_NEW_IN_DOT_4
#define LLDB_API_DEPRECATED_IN_DOT_4
#define LLDB_API_NEW_IN_DOT_5
#define LLDB_API_DEPRECATED_IN_DOT_5
#define LLDB_API_NEW_IN_DOT_6
#define LLDB_API_DEPRECATED_IN_DOT_6
#define LLDB_API_NEW_IN_DOT_7
#define LLDB_API_DEPRECATED_IN_DOT_7
#define LLDB_API_NEW_IN_DOT_8
#define LLDB_API_DEPRECATED_IN_DOT_8
#define LLDB_API_NEW_IN_DOT_9
#define LLDB_API_DEPRECATED_IN_DOT_9
#define LLDB_API_NEW_IN_DOT_10
#define LLDB_API_DEPRECATED_IN_DOT_10
#define LLDB_API_NEW_IN_DOT_11
#define LLDB_API_DEPRECATED_IN_DOT_11
#define LLDB_API_NEW_IN_DOT_12
#define LLDB_API_DEPRECATED_IN_DOT_12
#define LLDB_API_NEW_IN_DOT_13
#define LLDB_API_DEPRECATED_IN_DOT_13
#define LLDB_API_NEW_IN_DOT_14
#define LLDB_API_DEPRECATED_IN_DOT_14
#define LLDB_API_NEW_IN_DOT_15
#define LLDB_API_DEPRECATED_IN_DOT_15
#define LLDB_API_NEW_IN_DOT_16
#define LLDB_API_DEPRECATED_IN_DOT_16
#define LLDB_API_NEW_IN_DOT_17
#define LLDB_API_DEPRECATED_IN_DOT_17
#define LLDB_API_NEW_IN_DOT_18
#define LLDB_API_DEPRECATED_IN_DOT_18
#define LLDB_API_NEW_IN_DOT_19
#define LLDB_API_DEPRECATED_IN_DOT_19
#define LLDB_API_NEW_IN_DOT_20
#define LLDB_API_DEPRECATED_IN_DOT_20
#define LLDB_API_NEW_IN_DOT_21
#define LLDB_API_DEPRECATED_IN_DOT_21
#define LLDB_API_NEW_IN_DOT_22
#define LLDB_API_DEPRECATED_IN_DOT_22
#define LLDB_API_NEW_IN_DOT_23
#define LLDB_API_DEPRECATED_IN_DOT_23
#define LLDB_API_NEW_IN_DOT_24
#define LLDB_API_DEPRECATED_IN_DOT_24
#define LLDB_API_NEW_IN_DOT_25
#define LLDB_API_DEPRECATED_IN_DOT_25
#define LLDB_API_NEW_IN_DOT_26
#define LLDB_API_DEPRECATED_IN_DOT_26
#define LLDB_API_NEW_IN_DOT_27
#define LLDB_API_DEPRECATED_IN_DOT_27
#define LLDB_API_NEW_IN_DOT_28
#define LLDB_API_DEPRECATED_IN_DOT_28
#define LLDB_API_NEW_IN_DOT_29
#define LLDB_API_DEPRECATED_IN_DOT_29
#define LLDB_API_NEW_IN_DOT_30
#define LLDB_API_DEPRECATED_IN_DOT_30
#define LLDB_API_NEW_IN_DOT_31
#define LLDB_API_DEPRECATED_IN_DOT_31
#define LLDB_API_NEW_IN_DOT_32
#define LLDB_API_DEPRECATED_IN_DOT_32
#define LLDB_API_NEW_IN_DOT_33
#define LLDB_API_DEPRECATED_IN_DOT_33
#define LLDB_API_NEW_IN_DOT_34
#define LLDB_API_DEPRECATED_IN_DOT_34
#define LLDB_API_NEW_IN_DOT_35
#define LLDB_API_DEPRECATED_IN_DOT_35
#define LLDB_API_NEW_IN_DOT_36
#define LLDB_API_DEPRECATED_IN_DOT_36
#define LLDB_API_NEW_IN_DOT_37
#define LLDB_API_DEPRECATED_IN_DOT_37
#define LLDB_API_NEW_IN_DOT_38
#define LLDB_API_DEPRECATED_IN_DOT_38
#define LLDB_API_NEW_IN_DOT_39
#define LLDB_API_DEPRECATED_IN_DOT_39
#define LLDB_API_NEW_IN_DOT_40
#define LLDB_API_DEPRECATED_IN_DOT_40
#define LLDB_API_NEW_IN_DOT_41
#define LLDB_API_DEPRECATED_IN_DOT_41
#define LLDB_API_NEW_IN_DOT_42
#define LLDB_API_DEPRECATED_IN_DOT_42
#define LLDB_API_NEW_IN_DOT_43
#define LLDB_API_DEPRECATED_IN_DOT_43
#define LLDB_API_NEW_IN_DOT_44
#define LLDB_API_DEPRECATED_IN_DOT_44
#define LLDB_API_NEW_IN_DOT_45
#define LLDB_API_DEPRECATED_IN_DOT_45
#define LLDB_API_NEW_IN_DOT_46
#define LLDB_API_DEPRECATED_IN_DOT_46
#define LLDB_API_NEW_IN_DOT_47
#define LLDB_API_DEPRECATED_IN_DOT_47
#define LLDB_API_NEW_IN_DOT_48
#define LLDB_API_DEPRECATED_IN_DOT_48
#define LLDB_API_NEW_IN_DOT_49
#define LLDB_API_DEPRECATED_IN_DOT_49
#define LLDB_API_NEW_IN_DOT_50
#define LLDB_API_DEPRECATED_IN_DOT_50
#define LLDB_API_NEW_IN_DOT_51
#define LLDB_API_DEPRECATED_IN_DOT_51
#define LLDB_API_NEW_IN_DOT_52
#define LLDB_API_DEPRECATED_IN_DOT_52
#define LLDB_API_NEW_IN_DOT_53
#define LLDB_API_DEPRECATED_IN_DOT_53
#define LLDB_API_NEW_IN_DOT_54
#define LLDB_API_DEPRECATED_IN_DOT_54
#define LLDB_API_NEW_IN_DOT_55
#define LLDB_API_DEPRECATED_IN_DOT_55
#define LLDB_API_NEW_IN_DOT_56
#define LLDB_API_DEPRECATED_IN_DOT_56
#define LLDB_API_NEW_IN_DOT_57
#define LLDB_API_DEPRECATED_IN_DOT_57
#define LLDB_API_NEW_IN_DOT_58
#define LLDB_API_DEPRECATED_IN_DOT_58
#define LLDB_API_NEW_IN_DOT_59
#define LLDB_API_DEPRECATED_IN_DOT_59
#define LLDB_API_NEW_IN_DOT_60
#define LLDB_API_DEPRECATED_IN_DOT_60
#define LLDB_API_NEW_IN_DOT_61
#define LLDB_API_DEPRECATED_IN_DOT_61
#define LLDB_API_NEW_IN_DOT_62
#define LLDB_API_DEPRECATED_IN_DOT_62
#define LLDB_API_NEW_IN_DOT_63
#define LLDB_API_DEPRECATED_IN_DOT_63
#define LLDB_API_NEW_IN_DOT_64
#define LLDB_API_DEPRECATED_IN_DOT_64
#define LLDB_API_NEW_IN_DOT_65
#define LLDB_API_DEPRECATED_IN_DOT_65
#define LLDB_API_NEW_IN_DOT_66
#define LLDB_API_DEPRECATED_IN_DOT_66
#define LLDB_API_NEW_IN_DOT_67
#define LLDB_API_DEPRECATED_IN_DOT_67
#define LLDB_API_NEW_IN_DOT_68
#define LLDB_API_DEPRECATED_IN_DOT_68
#define LLDB_API_NEW_IN_DOT_69
#define LLDB_API_DEPRECATED_IN_DOT_69
#define LLDB_API_NEW_IN_DOT_70
#define LLDB_API_DEPRECATED_IN_DOT_70
#define LLDB_API_NEW_IN_DOT_71
#define LLDB_API_DEPRECATED_IN_DOT_71
#define LLDB_API_NEW_IN_DOT_72
#define LLDB_API_DEPRECATED_IN_DOT_72
#define LLDB_API_NEW_IN_DOT_73
#define LLDB_API_DEPRECATED_IN_DOT_73
#define LLDB_API_NEW_IN_DOT_74
#define LLDB_API_DEPRECATED_IN_DOT_74
#define LLDB_API_NEW_IN_DOT_75
#define LLDB_API_DEPRECATED_IN_DOT_75
#define LLDB_API_NEW_IN_DOT_76
#define LLDB_API_DEPRECATED_IN_DOT_76
#define LLDB_API_NEW_IN_DOT_77
#define LLDB_API_DEPRECATED_IN_DOT_77
#define LLDB_API_NEW_IN_DOT_78
#define LLDB_API_DEPRECATED_IN_DOT_78
#define LLDB_API_NEW_IN_DOT_79
#define LLDB_API_DEPRECATED_IN_DOT_79
#define LLDB_API_NEW_IN_DOT_80
#define LLDB_API_DEPRECATED_IN_DOT_80
#define LLDB_API_NEW_IN_DOT_81
#define LLDB_API_DEPRECATED_IN_DOT_81
#define LLDB_API_NEW_IN_DOT_82
#define LLDB_API_DEPRECATED_IN_DOT_82
#define LLDB_API_NEW_IN_DOT_83
#define LLDB_API_DEPRECATED_IN_DOT_83
#define LLDB_API_NEW_IN_DOT_84
#define LLDB_API_DEPRECATED_IN_DOT_84
#define LLDB_API_NEW_IN_DOT_85
#define LLDB_API_DEPRECATED_IN_DOT_85
#define LLDB_API_NEW_IN_DOT_86
#define LLDB_API_DEPRECATED_IN_DOT_86
#define LLDB_API_NEW_IN_DOT_87
#define LLDB_API_DEPRECATED_IN_DOT_87
#define LLDB_API_NEW_IN_DOT_88
#define LLDB_API_DEPRECATED_IN_DOT_88
#define LLDB_API_NEW_IN_DOT_89
#define LLDB_API_DEPRECATED_IN_DOT_89
#define LLDB_API_NEW_IN_DOT_90
#define LLDB_API_DEPRECATED_IN_DOT_90
#define LLDB_API_NEW_IN_DOT_91
#define LLDB_API_DEPRECATED_IN_DOT_91
#define LLDB_API_NEW_IN_DOT_92
#define LLDB_API_DEPRECATED_IN_DOT_92
#define LLDB_API_NEW_IN_DOT_93
#define LLDB_API_DEPRECATED_IN_DOT_93
#define LLDB_API_NEW_IN_DOT_94
#define LLDB_API_DEPRECATED_IN_DOT_94
#define LLDB_API_NEW_IN_DOT_95
#define LLDB_API_DEPRECATED_IN_DOT_95
#define LLDB_API_NEW_IN_DOT_96
#define LLDB_API_DEPRECATED_IN_DOT_96
#define LLDB_API_NEW_IN_DOT_97
#define LLDB_API_DEPRECATED_IN_DOT_97
#define LLDB_API_NEW_IN_DOT_98
#define LLDB_API_DEPRECATED_IN_DOT_98
#define LLDB_API_NEW_IN_DOT_99
#define LLDB_API_DEPRECATED_IN_DOT_99
#endif // defined(LLDB_CHECK_API_VERSIONING) &&
       // defined(LLDB_API_MAJOR_VERSION_WANTED) &&
       // defined(LLDB_API_MINOR_VERSION_WANTED) && defined
       // (LLDB_API_MAJOR_VERSION)

#endif // LLDB_LLDB_VERSIONING_H
