/**
 * Author......: See docs/credits.txt
 * License.....: MIT
 */

#include "common.h"
#include "types.h"
#include "modules.h"
#include "bitops.h"
#include "convert.h"
#include "shared.h"
#include "memory.h"

static const u32   ATTACK_EXEC    = ATTACK_EXEC_OUTSIDE_KERNEL;
static const u32   DGST_POS0      = 0;
static const u32   DGST_POS1      = 1;
static const u32   DGST_POS2      = 2;
static const u32   DGST_POS3      = 3;
static const u32   DGST_SIZE      = DGST_SIZE_4_16;
static const u32   HASH_CATEGORY  = HASH_CATEGORY_FDE;
static const char *HASH_NAME      = "LUKS v1 SHA-1 + AES";
static const u64   KERN_TYPE      = 14611; // old kernel used here
static const u32   OPTI_TYPE      = OPTI_TYPE_ZERO_BYTE
                                  | OPTI_TYPE_SLOW_HASH_SIMD_LOOP;
static const u64   OPTS_TYPE      = OPTS_TYPE_STOCK_MODULE
                                  | OPTS_TYPE_PT_GENERATE_LE;
static const u32   SALT_TYPE      = SALT_TYPE_EMBEDDED;
static const char *ST_PASS        = "hashcat";
static const char *ST_HASH        = "$luks$1$sha1$aes$cbc-essiv:sha256$128$151351$fe93ac0ab67cea12186b30980dd015a5652eec1ae0639aa34629c9e49118d235$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$e5c48d307b4b444184eba144844e053086000b83b9894620633b43da0b3fb601fd8d7d2a81cde7f941b9119d5f1e1caae59bfa236bc8652efef93219f8ae492e4b6a8f7ea8da084d93a7721e9baa58e2700c2606eb0af8adb8373a919238dc7bbde726bced1d58db9fae906d4498deb8e91f44a1c42fbb112d9cbe172952685c3ea4b2735b17b2180b330a2c40b27ae1464c62e774a10be913ee7c131f2324e9890716f32401d77fcaa12ad2e07ce5441b5bcbd6e85c005762759e64c88cbbc97d50fd11b2aed6f3a714f2e6f3d1b4fb6a71b744dabb28e20993810bb474726b77526bc8dd30c75e556405f615eb63acabc56fa97c5a19ed5c524286884bbdb1a22680fa35e74248cb44e646c28833cbd3eb0ce7478425aee76af39360327fe3f2f55db3823404cd8bcad8d6546eed5d6bc7921e810b34ed399cd15780f48f2a3cd2d41113c1b1c1961bb2810bf4fb9053838b3d7978c254ce9c4b7dae92f5341e1418756d0aac510d1e66af3a8d87155ab15a606084e54288ae2548bba0478d80f8af40893e9059d4a870747a8a902063f7878d1cc4703ce4c920edf438ab98e4535f5c2e5bff9f8e9e82ddd3f8d639ecaeab44a1a30a55a59e2a0b208f4cc0a54c1f10c2d20e7fe786ccbdae541f9dc1299451a51974968fdb8281c21daeccfe426666fe02cd10e54f0f5d7d0962b563d7b882d7e125d16ea871e59c995154";

u32         module_attack_exec    (MAYBE_UNUSED const hashconfig_t *hashconfig, MAYBE_UNUSED const user_options_t *user_options, MAYBE_UNUSED const user_options_extra_t *user_options_extra) { return ATTACK_EXEC;     }
u32         module_dgst_pos0      (MAYBE_UNUSED const hashconfig_t *hashconfig, MAYBE_UNUSED const user_options_t *user_options, MAYBE_UNUSED const user_options_extra_t *user_options_extra) { return DGST_POS0;       }
u32         module_dgst_pos1      (MAYBE_UNUSED const hashconfig_t *hashconfig, MAYBE_UNUSED const user_options_t *user_options, MAYBE_UNUSED const user_options_extra_t *user_options_extra) { return DGST_POS1;       }
u32         module_dgst_pos2      (MAYBE_UNUSED const hashconfig_t *hashconfig, MAYBE_UNUSED const user_options_t *user_options, MAYBE_UNUSED const user_options_extra_t *user_options_extra) { return DGST_POS2;       }
u32         module_dgst_pos3      (MAYBE_UNUSED const hashconfig_t *hashconfig, MAYBE_UNUSED const user_options_t *user_options, MAYBE_UNUSED const user_options_extra_t *user_options_extra) { return DGST_POS3;       }
u32         module_dgst_size      (MAYBE_UNUSED const hashconfig_t *hashconfig, MAYBE_UNUSED const user_options_t *user_options, MAYBE_UNUSED const user_options_extra_t *user_options_extra) { return DGST_SIZE;       }
u32         module_hash_category  (MAYBE_UNUSED const hashconfig_t *hashconfig, MAYBE_UNUSED const user_options_t *user_options, MAYBE_UNUSED const user_options_extra_t *user_options_extra) { return HASH_CATEGORY;   }
const char *module_hash_name      (MAYBE_UNUSED const hashconfig_t *hashconfig, MAYBE_UNUSED const user_options_t *user_options, MAYBE_UNUSED const user_options_extra_t *user_options_extra) { return HASH_NAME;       }
u64         module_kern_type      (MAYBE_UNUSED const hashconfig_t *hashconfig, MAYBE_UNUSED const user_options_t *user_options, MAYBE_UNUSED const user_options_extra_t *user_options_extra) { return KERN_TYPE;       }
u32         module_opti_type      (MAYBE_UNUSED const hashconfig_t *hashconfig, MAYBE_UNUSED const user_options_t *user_options, MAYBE_UNUSED const user_options_extra_t *user_options_extra) { return OPTI_TYPE;       }
u64         module_opts_type      (MAYBE_UNUSED const hashconfig_t *hashconfig, MAYBE_UNUSED const user_options_t *user_options, MAYBE_UNUSED const user_options_extra_t *user_options_extra) { return OPTS_TYPE;       }
u32         module_salt_type      (MAYBE_UNUSED const hashconfig_t *hashconfig, MAYBE_UNUSED const user_options_t *user_options, MAYBE_UNUSED const user_options_extra_t *user_options_extra) { return SALT_TYPE;       }
const char *module_st_hash        (MAYBE_UNUSED const hashconfig_t *hashconfig, MAYBE_UNUSED const user_options_t *user_options, MAYBE_UNUSED const user_options_extra_t *user_options_extra) { return ST_HASH;         }
const char *module_st_pass        (MAYBE_UNUSED const hashconfig_t *hashconfig, MAYBE_UNUSED const user_options_t *user_options, MAYBE_UNUSED const user_options_extra_t *user_options_extra) { return ST_PASS;         }

#define LUKS_STRIPES        (                                   4000)
#define LUKS_SALT_LEN       (                                     32)
#define LUKS_SALT_HEX_LEN   (                      LUKS_SALT_LEN * 2)
#define LUKS_CT_LEN         (                                    512)
#define LUKS_CT_HEX_LEN     (                        LUKS_CT_LEN * 2)
#define LUKS_AF_MIN_LEN     (HC_LUKS_KEY_SIZE_128 / 8 * LUKS_STRIPES)
#define LUKS_AF_MIN_HEX_LEN (                    LUKS_AF_MIN_LEN * 2)
#define LUKS_AF_MAX_LEN     (HC_LUKS_KEY_SIZE_512 / 8 * LUKS_STRIPES)
#define LUKS_AF_MAX_HEX_LEN (                    LUKS_AF_MAX_LEN * 2)

typedef enum hc_luks_hash_type
{
  HC_LUKS_HASH_TYPE_SHA1      = 1,
  HC_LUKS_HASH_TYPE_SHA256    = 2,
  HC_LUKS_HASH_TYPE_SHA512    = 3,
  HC_LUKS_HASH_TYPE_RIPEMD160 = 4,
  HC_LUKS_HASH_TYPE_WHIRLPOOL = 5,

} hc_luks_hash_type_t;

typedef enum hc_luks_key_size
{
  HC_LUKS_KEY_SIZE_128 = 128,
  HC_LUKS_KEY_SIZE_256 = 256,
  HC_LUKS_KEY_SIZE_512 = 512,

} hc_luks_key_size_t;

typedef enum hc_luks_cipher_type
{
  HC_LUKS_CIPHER_TYPE_AES     = 1,
  HC_LUKS_CIPHER_TYPE_SERPENT = 2,
  HC_LUKS_CIPHER_TYPE_TWOFISH = 3,

} hc_luks_cipher_type_t;

typedef enum hc_luks_cipher_mode
{
  HC_LUKS_CIPHER_MODE_CBC_ESSIV_SHA256 = 1,
  HC_LUKS_CIPHER_MODE_CBC_PLAIN        = 2,
  HC_LUKS_CIPHER_MODE_CBC_PLAIN64      = 3,
  HC_LUKS_CIPHER_MODE_XTS_PLAIN        = 4,
  HC_LUKS_CIPHER_MODE_XTS_PLAIN64      = 5,

} hc_luks_cipher_mode_t;

typedef struct luks
{
  int hash_type;   // hc_luks_hash_type_t
  int key_size;    // hc_luks_key_size_t
  int cipher_type; // hc_luks_cipher_type_t
  int cipher_mode; // hc_luks_cipher_mode_t

  u32 ct_buf[LUKS_CT_LEN / 4];

  u32 af_buf[LUKS_AF_MAX_LEN / 4];
  u32 af_len;

} luks_t;

typedef struct luks_tmp
{
  u32 ipad32[8];
  u64 ipad64[8];

  u32 opad32[8];
  u64 opad64[8];

  u32 dgst32[32];
  u64 dgst64[16];

  u32 out32[32];
  u64 out64[16];

} luks_tmp_t;

static const char *SIGNATURE_LUKS = "$luks$1$sha1$aes$";

u64 module_esalt_size (MAYBE_UNUSED const hashconfig_t *hashconfig, MAYBE_UNUSED const user_options_t *user_options, MAYBE_UNUSED const user_options_extra_t *user_options_extra)
{
  const u64 esalt_size = (const u64) sizeof (luks_t);

  return esalt_size;
}

u64 module_tmp_size (MAYBE_UNUSED const hashconfig_t *hashconfig, MAYBE_UNUSED const user_options_t *user_options, MAYBE_UNUSED const user_options_extra_t *user_options_extra)
{
  const u64 tmp_size = (const u64) sizeof (luks_tmp_t);

  return tmp_size;
}

u32 module_pw_max (MAYBE_UNUSED const hashconfig_t *hashconfig, MAYBE_UNUSED const user_options_t *user_options, MAYBE_UNUSED const user_options_extra_t *user_options_extra)
{
  // this overrides the reductions of PW_MAX in case optimized kernel is selected
  // IOW, even in optimized kernel mode it support length 256

  const u32 pw_max = PW_MAX;

  return pw_max;
}

int module_hash_decode (MAYBE_UNUSED const hashconfig_t *hashconfig, MAYBE_UNUSED void *digest_buf, MAYBE_UNUSED salt_t *salt, MAYBE_UNUSED void *esalt_buf, MAYBE_UNUSED void *hook_salt_buf, MAYBE_UNUSED hashinfo_t *hash_info, const char *line_buf, MAYBE_UNUSED const int line_len)
{
  luks_t *luks = (luks_t *) esalt_buf;

  hc_token_t token;

  token.token_cnt  = 7;

  token.signatures_cnt    = 1;
  token.signatures_buf[0] = SIGNATURE_LUKS;

  // signature with hash and cipher type
  token.len[0]     = 17;
  token.attr[0]    = TOKEN_ATTR_FIXED_LENGTH
                   | TOKEN_ATTR_VERIFY_SIGNATURE;

  // cipher mode
  token.sep[1]     = '$';
  token.len_min[1] = 9;
  token.len_max[1] = 16;
  token.attr[1]    = TOKEN_ATTR_VERIFY_LENGTH;

  // key size
  token.sep[2]     = '$';
  token.len_min[2] = 3;
  token.len_max[2] = 3;
  token.attr[2]    = TOKEN_ATTR_VERIFY_LENGTH
                   | TOKEN_ATTR_VERIFY_DIGIT;

  // iter
  token.sep[3]     = '$';
  token.len_min[3] = 1;
  token.len_max[3] = 10;
  token.attr[3]    = TOKEN_ATTR_VERIFY_LENGTH
                   | TOKEN_ATTR_VERIFY_DIGIT;

  // salt
  token.sep[4]     = '$';
  token.len_min[4] = LUKS_SALT_HEX_LEN;
  token.len_max[4] = LUKS_SALT_HEX_LEN;
  token.attr[4]    = TOKEN_ATTR_VERIFY_LENGTH
                   | TOKEN_ATTR_VERIFY_HEX;

  // af
  token.sep[5]     = '$';
  token.len_min[5] = LUKS_AF_MIN_HEX_LEN;
  token.len_max[5] = LUKS_AF_MAX_HEX_LEN;
  token.attr[5]    = TOKEN_ATTR_VERIFY_LENGTH
                   | TOKEN_ATTR_VERIFY_HEX;

  // ct
  token.sep[6]     = '$';
  token.len_min[6] = LUKS_CT_HEX_LEN;
  token.len_max[6] = LUKS_CT_HEX_LEN;
  token.attr[6]    = TOKEN_ATTR_VERIFY_LENGTH
                   | TOKEN_ATTR_VERIFY_HEX;

  const int rc_tokenizer = input_tokenizer ((const u8 *) line_buf, line_len, &token);

  if (rc_tokenizer != PARSER_OK) return (rc_tokenizer);

  // hash type

  luks->hash_type = HC_LUKS_HASH_TYPE_SHA1;

  // cipher type

  luks->cipher_type = HC_LUKS_CIPHER_TYPE_AES;

  // cipher mode

  const u8 *cipher_mode_pos = token.buf[1];

  if (strncmp ((const char *) cipher_mode_pos, "cbc-essiv:sha256", 16) == 0)
  {
    luks->cipher_mode = HC_LUKS_CIPHER_MODE_CBC_ESSIV_SHA256;
  }
  else if (strncmp ((const char *) cipher_mode_pos, "cbc-plain",    9) == 0)
  {
    luks->cipher_mode = HC_LUKS_CIPHER_MODE_CBC_PLAIN;
  }
  else if (strncmp ((const char *) cipher_mode_pos, "cbc-plain64", 11) == 0)
  {
    luks->cipher_mode = HC_LUKS_CIPHER_MODE_CBC_PLAIN64;
  }
  else if (strncmp ((const char *) cipher_mode_pos, "xts-plain",    9) == 0)
  {
    luks->cipher_mode = HC_LUKS_CIPHER_MODE_XTS_PLAIN;
  }
  else if (strncmp ((const char *) cipher_mode_pos, "xts-plain64", 11) == 0)
  {
    luks->cipher_mode = HC_LUKS_CIPHER_MODE_XTS_PLAIN64;
  }
  else
  {
    return (PARSER_LUKS_CIPHER_MODE);
  }

  // key size

  const u8 *key_size_pos = token.buf[2];

  const u32 key_size = hc_strtoul ((const char *) key_size_pos, NULL, 10);

  if (key_size == 128)
  {
    luks->key_size = HC_LUKS_KEY_SIZE_128;
  }
  else if (key_size == 256)
  {
    luks->key_size = HC_LUKS_KEY_SIZE_256;
  }
  else if (key_size == 512)
  {
    luks->key_size = HC_LUKS_KEY_SIZE_512;
  }
  else
  {
    return (PARSER_LUKS_KEY_SIZE);
  }

  // salt

  const u8 *salt_pos = token.buf[4];

  salt->salt_len = hex_decode (salt_pos, LUKS_SALT_HEX_LEN, (u8 *) salt->salt_buf);

  // iter

  const u8 *iter_pos = token.buf[3];

  const u32 iter = hc_strtoul ((const char *) iter_pos, NULL, 10);

  salt->salt_iter = iter - 1;

  // af

  u32 af_len = token.len[5] / 2;

  if (af_len != (key_size / 8) * LUKS_STRIPES) return (PARSER_LUKS_FILE_SIZE);

  const u8 *af_pos = token.buf[5];

  luks->af_len = hex_decode (af_pos, token.len[5], (u8 *) luks->af_buf);

  // ct

  u32 ct_len = token.len[6] / 2;

  if (ct_len != LUKS_CT_LEN) return (PARSER_LUKS_FILE_SIZE);

  const u8 *ct_pos = token.buf[6];

  hex_decode (ct_pos, LUKS_CT_HEX_LEN, (u8 *) luks->ct_buf);

  return (PARSER_OK);
}

int module_hash_encode (MAYBE_UNUSED const hashconfig_t *hashconfig, MAYBE_UNUSED const void *digest_buf, MAYBE_UNUSED const salt_t *salt, MAYBE_UNUSED const void *esalt_buf, MAYBE_UNUSED const void *hook_salt_buf, MAYBE_UNUSED const hashinfo_t *hash_info, char *line_buf, MAYBE_UNUSED const int line_size)
{
  luks_t *luks = (luks_t *) esalt_buf;

  // cipher mode

  char *cipher_mode = "";

  switch (luks->cipher_mode)
  {
    case HC_LUKS_CIPHER_MODE_CBC_ESSIV_SHA256: cipher_mode = "cbc-essiv:sha256"; break;
    case HC_LUKS_CIPHER_MODE_CBC_PLAIN:        cipher_mode = "cbc-plain";        break;
    case HC_LUKS_CIPHER_MODE_CBC_PLAIN64:      cipher_mode = "cbc-plain64";      break;
    case HC_LUKS_CIPHER_MODE_XTS_PLAIN:        cipher_mode = "xts-plain";        break;
    case HC_LUKS_CIPHER_MODE_XTS_PLAIN64:      cipher_mode = "xts-plain64";      break;
    default:                                                                     return 0;
  }

  // key size

  u32 key_size = 0;

  switch (luks->key_size)
  {
    case HC_LUKS_KEY_SIZE_128: key_size = 128; break;
    case HC_LUKS_KEY_SIZE_256: key_size = 256; break;
    case HC_LUKS_KEY_SIZE_512: key_size = 512; break;
    default:                                   return 0;
  }

  // iterations

  u32 iterations = salt->salt_iter + 1;

  // salt

  char salt_buf[LUKS_SALT_HEX_LEN + 1] = { 0 };

  hex_encode ((const u8 *) salt->salt_buf, LUKS_SALT_LEN, (u8 *) salt_buf);

  // af

  char af_buf[LUKS_AF_MAX_HEX_LEN + 1] = { 0 };

  hex_encode ((const u8 *) luks->af_buf, luks->af_len, (u8 *) af_buf);

  // ct

  char ct_buf[LUKS_CT_HEX_LEN + 1] = { 0 };

  hex_encode ((const u8 *) luks->ct_buf, LUKS_CT_LEN, (u8 *) ct_buf);

  // output

  int line_len = snprintf (line_buf, line_size, "%s%s$%u$%u$%s$%s$%s",
    SIGNATURE_LUKS,
    cipher_mode,
    key_size,
    iterations,
    salt_buf,
    af_buf,
    ct_buf);

  return line_len;
}

void module_init (module_ctx_t *module_ctx)
{
  module_ctx->module_context_size             = MODULE_CONTEXT_SIZE_CURRENT;
  module_ctx->module_interface_version        = MODULE_INTERFACE_VERSION_CURRENT;

  module_ctx->module_attack_exec              = module_attack_exec;
  module_ctx->module_benchmark_esalt          = MODULE_DEFAULT;
  module_ctx->module_benchmark_hook_salt      = MODULE_DEFAULT;
  module_ctx->module_benchmark_mask           = MODULE_DEFAULT;
  module_ctx->module_benchmark_charset        = MODULE_DEFAULT;
  module_ctx->module_benchmark_salt           = MODULE_DEFAULT;
  module_ctx->module_build_plain_postprocess  = MODULE_DEFAULT;
  module_ctx->module_deep_comp_kernel         = MODULE_DEFAULT;
  module_ctx->module_deprecated_notice        = MODULE_DEFAULT;
  module_ctx->module_dgst_pos0                = module_dgst_pos0;
  module_ctx->module_dgst_pos1                = module_dgst_pos1;
  module_ctx->module_dgst_pos2                = module_dgst_pos2;
  module_ctx->module_dgst_pos3                = module_dgst_pos3;
  module_ctx->module_dgst_size                = module_dgst_size;
  module_ctx->module_dictstat_disable         = MODULE_DEFAULT;
  module_ctx->module_esalt_size               = module_esalt_size;
  module_ctx->module_extra_buffer_size        = MODULE_DEFAULT;
  module_ctx->module_extra_tmp_size           = MODULE_DEFAULT;
  module_ctx->module_extra_tuningdb_block     = MODULE_DEFAULT;
  module_ctx->module_forced_outfile_format    = MODULE_DEFAULT;
  module_ctx->module_hash_binary_count        = MODULE_DEFAULT;
  module_ctx->module_hash_binary_parse        = MODULE_DEFAULT;
  module_ctx->module_hash_binary_save         = MODULE_DEFAULT;
  module_ctx->module_hash_decode_postprocess  = MODULE_DEFAULT;
  module_ctx->module_hash_decode_potfile      = MODULE_DEFAULT;
  module_ctx->module_hash_decode_zero_hash    = MODULE_DEFAULT;
  module_ctx->module_hash_decode              = module_hash_decode;
  module_ctx->module_hash_encode_status       = MODULE_DEFAULT;
  module_ctx->module_hash_encode_potfile      = MODULE_DEFAULT;
  module_ctx->module_hash_encode              = module_hash_encode;
  module_ctx->module_hash_init_selftest       = MODULE_DEFAULT;
  module_ctx->module_hash_mode                = MODULE_DEFAULT;
  module_ctx->module_hash_category            = module_hash_category;
  module_ctx->module_hash_name                = module_hash_name;
  module_ctx->module_hashes_count_min         = MODULE_DEFAULT;
  module_ctx->module_hashes_count_max         = MODULE_DEFAULT;
  module_ctx->module_hlfmt_disable            = MODULE_DEFAULT;
  module_ctx->module_hook_extra_param_size    = MODULE_DEFAULT;
  module_ctx->module_hook_extra_param_init    = MODULE_DEFAULT;
  module_ctx->module_hook_extra_param_term    = MODULE_DEFAULT;
  module_ctx->module_hook12                   = MODULE_DEFAULT;
  module_ctx->module_hook23                   = MODULE_DEFAULT;
  module_ctx->module_hook_salt_size           = MODULE_DEFAULT;
  module_ctx->module_hook_size                = MODULE_DEFAULT;
  module_ctx->module_jit_build_options        = MODULE_DEFAULT;
  module_ctx->module_jit_cache_disable        = MODULE_DEFAULT;
  module_ctx->module_kernel_accel_max         = MODULE_DEFAULT;
  module_ctx->module_kernel_accel_min         = MODULE_DEFAULT;
  module_ctx->module_kernel_loops_max         = MODULE_DEFAULT;
  module_ctx->module_kernel_loops_min         = MODULE_DEFAULT;
  module_ctx->module_kernel_threads_max       = MODULE_DEFAULT;
  module_ctx->module_kernel_threads_min       = MODULE_DEFAULT;
  module_ctx->module_kern_type                = module_kern_type;
  module_ctx->module_kern_type_dynamic        = MODULE_DEFAULT;
  module_ctx->module_opti_type                = module_opti_type;
  module_ctx->module_opts_type                = module_opts_type;
  module_ctx->module_outfile_check_disable    = MODULE_DEFAULT;
  module_ctx->module_outfile_check_nocomp     = MODULE_DEFAULT;
  module_ctx->module_potfile_custom_check     = MODULE_DEFAULT;
  module_ctx->module_potfile_disable          = MODULE_DEFAULT;
  module_ctx->module_potfile_keep_all_hashes  = MODULE_DEFAULT;
  module_ctx->module_pwdump_column            = MODULE_DEFAULT;
  module_ctx->module_pw_max                   = module_pw_max;
  module_ctx->module_pw_min                   = MODULE_DEFAULT;
  module_ctx->module_salt_max                 = MODULE_DEFAULT;
  module_ctx->module_salt_min                 = MODULE_DEFAULT;
  module_ctx->module_salt_type                = module_salt_type;
  module_ctx->module_separator                = MODULE_DEFAULT;
  module_ctx->module_st_hash                  = module_st_hash;
  module_ctx->module_st_pass                  = module_st_pass;
  module_ctx->module_tmp_size                 = module_tmp_size;
  module_ctx->module_unstable_warning         = MODULE_DEFAULT;
  module_ctx->module_warmup_disable           = MODULE_DEFAULT;
}
