import datetime
import sys
import uuid

from peewee import Model
from flask_peewee.utils import get_dictionary_from_model
from flask_peewee.utils import get_model_from_dictionary


class Serializer(object):
    date_format = '%Y-%m-%d'
    time_format = '%H:%M:%S'
    datetime_format = 'T'.join([date_format, time_format])
    use_iso8601 = True

    def convert_value(self, value):
        if isinstance(value, datetime.datetime):
            if self.use_iso8601:
                return value.isoformat()
            else:
                return value.strftime(self.datetime_format)
        elif isinstance(value, datetime.date):
            return value.strftime(self.date_format)
        elif isinstance(value, datetime.time):
            return value.strftime(self.time_format)
        elif isinstance(value, datetime.timedelta):
            return value.total_seconds()
        elif isinstance(value, Model):
            return value._pk
        elif isinstance(value, uuid.UUID):
            return str(value)
        else:
            return value

    def clean_data(self, data):
        for key, value in data.items():
            if isinstance(value, dict):
                self.clean_data(value)
            elif isinstance(value, (list, tuple)):
                data[key] = list(map(self.clean_data, value))
            else:
                data[key] = self.convert_value(value)
        return data

    def serialize_object(self, obj, fields=None, exclude=None):
        data = get_dictionary_from_model(obj, fields, exclude)
        return self.clean_data(data)


class Deserializer(object):
    def deserialize_object(self, model, data):
        return get_model_from_dictionary(model, data)
