/** @file monitor_model_key.h
 *  Uniquely identifies a monitor model using its manufacturer id,
 *  model name, and product code, as listed in the EDID.
 */

// Copyright (C) 2018-2024 Sanford Rockowitz <rockowitz@minsoft.com>
// SPDX-License-Identifier: GPL-2.0-or-later

#ifndef MONITOR_MODEL_KEY_H_
#define MONITOR_MODEL_KEY_H_

#include <inttypes.h>

#include "public/ddcutil_types.h"

#include "util/edid.h"

#define FIXUP_MODEL_NAME(_name) \
   for (int i=0; _name[i] && i < EDID_MODEL_NAME_FIELD_SIZE; i++) { \
      if (!isalnum(_name[i])) \
         _name[i] = '_'; \
   }


/** Identifies a monitor model */
typedef struct {
   char                mfg_id[DDCA_EDID_MFG_ID_FIELD_SIZE];
   char                model_name[DDCA_EDID_MODEL_NAME_FIELD_SIZE];
   uint16_t            product_code;
   bool                defined;
} Monitor_Model_Key;


Monitor_Model_Key
mmk_value(
      const char *   mfg_id,
      const char *   model_name,
      uint16_t       product_code);

Monitor_Model_Key
mmk_undefined_value();

Monitor_Model_Key
mmk_value_from_edid(Parsed_Edid * edid);

Monitor_Model_Key *
mmk_new(
      const char *   mfg_id,
      const char *   model_name,
      uint16_t       product_code);

Monitor_Model_Key *
mmk_new_from_edid(
      Parsed_Edid * edid);

Monitor_Model_Key
mmk_value_from_string(const char * sval);

Monitor_Model_Key *
mmk_new_from_value(Monitor_Model_Key mmk);

Monitor_Model_Key *
mmk_new_from_string(const char * s);

#ifdef UNUSED
Monitor_Model_Key *
monitor_model_key_undefined_new();
#endif

void
mmk_free(
      Monitor_Model_Key * model_id);

char *
mmk_model_id_string(
      const char *  mfg,
      const char *  model_name,
      uint16_t      product_code);

bool
monitor_model_key_eq(
      Monitor_Model_Key mmk1,
      Monitor_Model_Key mmk2);

#ifdef UNUSED
bool monitor_model_key_is_defined(Monitor_Model_Key mmk);
#endif

char *
mmk_string(
      Monitor_Model_Key * model_id);

char *
mmk_repr(Monitor_Model_Key mmk);

void
init_monitor_model_key();

#endif /* MONITOR_MODEL_KEY_H_ */
