/*
  This file is part of CDO. CDO is a collection of Operators to
  manipulate and analyse Climate model Data.

  Copyright (C) 2003-2020 Uwe Schulzweida, <uwe.schulzweida AT mpimet.mpg.de>
  See COPYING file for copying and redistribution conditions.

  This program is free software; you can redistribute it and/or modify
  it under the terms of the GNU General Public License as published by
  the Free Software Foundation; version 2 of the License.

  This program is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
  GNU General Public License for more details.
*/

/*
   This module contains the following operators:

      Intyear    intyear         Year interpolation
*/

#include <cdi.h>

#include "cdo_rlimit.h"
#include "functs.h"
#include "process_int.h"
#include "cdo_vlist.h"
#include "util_files.h"
#include "param_conversion.h"


void *
Intyear(void *process)
{
  int nrecs;

  cdoInitialize(process);

  operatorInputArg("years");

  const auto iyears = cdoArgvToInt(cdoGetOperArgv());
  const int nyears = iyears.size();

  cdo::set_numfiles(nyears + 8);

  std::vector<CdoStreamID> streamIDs(nyears);

  const auto streamID1 = cdoOpenRead(0);
  const auto streamID2 = cdoOpenRead(1);

  const auto vlistID1 = cdoStreamInqVlist(streamID1);
  const auto vlistID2 = cdoStreamInqVlist(streamID2);
  const auto vlistID3 = vlistDuplicate(vlistID1);

  vlistCompare(vlistID1, vlistID2, CMP_ALL);

  const auto gridsizemax = vlistGridsizeMax(vlistID1);
  Varray<double> array1(gridsizemax), array2(gridsizemax), array3(gridsizemax);

  const auto taxisID1 = vlistInqTaxis(vlistID1);
  const auto taxisID2 = vlistInqTaxis(vlistID2);
  const auto taxisID3 = taxisDuplicate(taxisID1);
  if (taxisHasBounds(taxisID3)) taxisDeleteBounds(taxisID3);
  vlistDefTaxis(vlistID3, taxisID3);

  char filename[8192];
  strcpy(filename, cdoGetObase());
  const int nchars = strlen(filename);

  auto refname = cdoGetStreamName(0);
  char filesuffix[32] = { 0 };
  cdoGenFileSuffix(filesuffix, sizeof(filesuffix), cdoInqFiletype(streamID1), vlistID1, refname);

  for (int iy = 0; iy < nyears; iy++)
    {
      sprintf(filename + nchars, "%04d", iyears[iy]);
      if (filesuffix[0]) sprintf(filename + nchars + 4, "%s", filesuffix);

      streamIDs[iy] = cdoOpenWrite(filename);
      cdoDefVlist(streamIDs[iy], vlistID3);
    }

  VarList varList1, varList2;
  varListInit(varList1, vlistID1);
  varListInit(varList2, vlistID2);

  int tsID = 0;
  while (true)
    {
      nrecs = cdoStreamInqTimestep(streamID1, tsID);
      if (nrecs == 0) break;
      nrecs = cdoStreamInqTimestep(streamID2, tsID);
      if (nrecs == 0) cdoAbort("Too few timesteps in second inputfile!");

      const auto vtime = taxisInqVtime(taxisID1);
      const auto vdate1 = taxisInqVdate(taxisID1);
      const auto year1 = vdate1 / 10000;
      const auto vdate2 = taxisInqVdate(taxisID2);
      const auto year2 = vdate2 / 10000;

      for (int iy = 0; iy < nyears; iy++)
        {
          if (iyears[iy] < year1 || iyears[iy] > year2)
            cdoAbort("Year %d out of bounds (first year %d; last year %d)!", iyears[iy], year1, year2);
          const auto vdate3 = vdate1 - year1 * 10000 + iyears[iy] * 10000;
          taxisDefVdate(taxisID3, vdate3);
          taxisDefVtime(taxisID3, vtime);
          cdoDefTimestep(streamIDs[iy], tsID);
        }

      for (int recID = 0; recID < nrecs; recID++)
        {
          int varID, levelID;
          cdoInqRecord(streamID1, &varID, &levelID);
          cdoInqRecord(streamID2, &varID, &levelID);

          size_t nmiss1, nmiss2, nmiss3;
          cdoReadRecord(streamID1, array1.data(), &nmiss1);
          cdoReadRecord(streamID2, array2.data(), &nmiss2);

          const auto gridsize = varList1[varID].gridsize;

          for (int iy = 0; iy < nyears; iy++)
            {
              const double fac1 = ((double) year2 - iyears[iy]) / (year2 - year1);
              const double fac2 = ((double) iyears[iy] - year1) / (year2 - year1);

              nmiss3 = 0;

              if (nmiss1 || nmiss2)
                {
                  const auto missval1 = varList1[varID].missval;
                  const auto missval2 = varList2[varID].missval;

                  for (size_t i = 0; i < gridsize; i++)
                    {
                      if (!DBL_IS_EQUAL(array1[i], missval1) && !DBL_IS_EQUAL(array2[i], missval2))
                        {
                          array3[i] = array1[i] * fac1 + array2[i] * fac2;
                        }
                      else
                        {
                          array3[i] = missval1;
                          nmiss3++;
                        }
                    }
                }
              else
                {
                  for (size_t i = 0; i < gridsize; i++) array3[i] = array1[i] * fac1 + array2[i] * fac2;
                }

              cdoDefRecord(streamIDs[iy], varID, levelID);
              cdoWriteRecord(streamIDs[iy], array3.data(), nmiss3);
            }
        }

      tsID++;
    }

  for (int iy = 0; iy < nyears; iy++) cdoStreamClose(streamIDs[iy]);

  cdoStreamClose(streamID2);
  cdoStreamClose(streamID1);

  cdoFinish();

  return nullptr;
}
