/* SPDX-License-Identifier: GPL-2.0-or-later
 * Copyright 2017 Blender Foundation. */

/** \file
 * \ingroup edgpencil
 */

#include "BLI_math.h"
#include "BLI_utildefines.h"

#include "DNA_gpencil_types.h"
#include "DNA_material_types.h"
#include "DNA_object_types.h"
#include "DNA_scene_types.h"

#include "BKE_context.h"
#include "BKE_gpencil.h"
#include "BKE_gpencil_geom.h"
#include "BKE_main.h"
#include "BKE_material.h"

#include "BLT_translation.h"

#include "DEG_depsgraph.h"

#include "ED_gpencil.h"

void ED_gpencil_stroke_init_data(bGPDstroke *gps,
                                 const float *array,
                                 const int totpoints,
                                 const float mat[4][4])
{
  for (int i = 0; i < totpoints; i++) {
    bGPDspoint *pt = &gps->points[i];
    const int x = GP_PRIM_DATABUF_SIZE * i;

    pt->x = array[x];
    pt->y = array[x + 1];
    pt->z = array[x + 2];
    mul_m4_v3(mat, &pt->x);

    pt->pressure = array[x + 3];
    pt->strength = array[x + 4];
  }
}

/* Definition of the most important info from a color */
typedef struct ColorTemplate {
  const char *name;
  float line[4];
  float fill[4];
} ColorTemplate;

/* Add color an ensure duplications (matched by name) */
static int gpencil_monkey_color(
    Main *bmain, Object *ob, const ColorTemplate *pct, bool stroke, bool fill)
{
  int index;
  Material *ma = BKE_gpencil_object_material_ensure_by_name(bmain, ob, DATA_(pct->name), &index);

  copy_v4_v4(ma->gp_style->stroke_rgba, pct->line);
  srgb_to_linearrgb_v4(ma->gp_style->stroke_rgba, ma->gp_style->stroke_rgba);

  copy_v4_v4(ma->gp_style->fill_rgba, pct->fill);
  srgb_to_linearrgb_v4(ma->gp_style->fill_rgba, ma->gp_style->fill_rgba);

  if (!stroke) {
    ma->gp_style->flag &= ~GP_MATERIAL_STROKE_SHOW;
  }

  if (!fill) {
    ma->gp_style->flag &= ~GP_MATERIAL_FILL_SHOW;
  }
  else {
    ma->gp_style->flag |= GP_MATERIAL_FILL_SHOW;
  }

  return index;
}

/* ***************************************************************** */
/* Monkey Geometry */
static const float data0[270 * GP_PRIM_DATABUF_SIZE] = {
    -0.4911f, 0.0f, -0.1781f, 0.267f, 0.362f, -0.5168f, 0.0f, -0.1806f, 0.31f,  0.407f,
    -0.5361f, 0.0f, -0.1817f, 0.38f,  0.439f, -0.5618f, 0.0f, -0.1829f, 0.433f, 0.458f,
    -0.5892f, 0.0f, -0.1827f, 0.471f, 0.5f,   -0.6193f, 0.0f, -0.1814f, 0.496f, 0.516f,
    -0.6499f, 0.0f, -0.1782f, 0.511f, 0.519f, -0.6808f, 0.0f, -0.1729f, 0.521f, 0.53f,
    -0.7107f, 0.0f, -0.1651f, 0.527f, 0.533f, -0.7404f, 0.0f, -0.1555f, 0.531f, 0.534f,
    -0.7698f, 0.0f, -0.1447f, 0.534f, 0.535f, -0.7993f, 0.0f, -0.1332f, 0.535f, 0.535f,
    -0.8289f, 0.0f, -0.1209f, 0.536f, 0.537f, -0.8586f, 0.0f, -0.109f,  0.536f, 0.537f,
    -0.8871f, 0.0f, -0.0973f, 0.536f, 0.537f, -0.9125f, 0.0f, -0.0838f, 0.535f, 0.535f,
    -0.9353f, 0.0f, -0.0688f, 0.534f, 0.534f, -0.9561f, 0.0f, -0.0525f, 0.534f, 0.534f,
    -0.9752f, 0.0f, -0.0346f, 0.533f, 0.534f, -0.9944f, 0.0f, -0.016f,  0.533f, 0.534f,
    -1.0148f, 0.0f, 0.0028f,  0.532f, 0.532f, -1.0348f, 0.0f, 0.0215f,  0.531f, 0.531f,
    -1.05f,   0.0f, 0.0407f,  0.531f, 0.531f, -1.0639f, 0.0f, 0.0613f,  0.532f, 0.532f,
    -1.0752f, 0.0f, 0.0838f,  0.535f, 0.535f, -1.0848f, 0.0f, 0.1082f,  0.54f,  0.54f,
    -1.0936f, 0.0f, 0.1346f,  0.542f, 0.542f, -1.1024f, 0.0f, 0.1639f,  0.543f, 0.543f,
    -1.1102f, 0.0f, 0.1953f,  0.543f, 0.543f, -1.1128f, 0.0f, 0.2277f,  0.546f, 0.546f,
    -1.1091f, 0.0f, 0.2579f,  0.549f, 0.549f, -1.1023f, 0.0f, 0.2849f,  0.549f, 0.549f,
    -1.0934f, 0.0f, 0.3086f,  0.549f, 0.549f, -1.0831f, 0.0f, 0.3285f,  0.549f, 0.549f,
    -1.0724f, 0.0f, 0.3451f,  0.551f, 0.551f, -1.0607f, 0.0f, 0.3594f,  0.553f, 0.553f,
    -1.0474f, 0.0f, 0.3713f,  0.554f, 0.554f, -1.031f,  0.0f, 0.3804f,  0.554f, 0.554f,
    -1.0108f, 0.0f, 0.3874f,  0.555f, 0.555f, -0.9862f, 0.0f, 0.3922f,  0.556f, 0.556f,
    -0.9568f, 0.0f, 0.3941f,  0.557f, 0.557f, -0.9243f, 0.0f, 0.3934f,  0.557f, 0.557f,
    -0.8897f, 0.0f, 0.3861f,  0.557f, 0.557f, -0.8556f, 0.0f, 0.3754f,  0.557f, 0.557f,
    -0.8237f, 0.0f, 0.3608f,  0.558f, 0.557f, -0.7982f, 0.0f, 0.344f,   0.558f, 0.558f,
    -0.7786f, 0.0f, 0.329f,   0.557f, 0.559f, -0.7633f, 0.0f, 0.3183f,  0.556f, 0.559f,
    -0.7498f, 0.0f, 0.3135f,  0.554f, 0.559f, -0.7374f, 0.0f, 0.3134f,  0.552f, 0.548f,
    -0.7261f, 0.0f, 0.3179f,  0.551f, 0.546f, -0.7146f, 0.0f, 0.3262f,  0.55f,  0.547f,
    -0.703f,  0.0f, 0.3395f,  0.549f, 0.547f, -0.692f,  0.0f, 0.3576f,  0.549f, 0.548f,
    -0.6831f, 0.0f, 0.3806f,  0.549f, 0.548f, -0.6748f, 0.0f, 0.4052f,  0.55f,  0.549f,
    -0.6648f, 0.0f, 0.4305f,  0.552f, 0.548f, -0.6527f, 0.0f, 0.4549f,  0.556f, 0.549f,
    -0.6375f, 0.0f, 0.4783f,  0.563f, 0.549f, -0.6195f, 0.0f, 0.5021f,  0.572f, 0.558f,
    -0.5985f, 0.0f, 0.5256f,  0.582f, 0.587f, -0.5775f, 0.0f, 0.5488f,  0.591f, 0.595f,
    -0.556f,  0.0f, 0.5715f,  0.597f, 0.598f, -0.5339f, 0.0f, 0.593f,   0.602f, 0.603f,
    -0.5119f, 0.0f, 0.613f,   0.605f, 0.606f, -0.4905f, 0.0f, 0.6312f,  0.607f, 0.607f,
    -0.4697f, 0.0f, 0.6474f,  0.609f, 0.607f, -0.4499f, 0.0f, 0.6613f,  0.612f, 0.611f,
    -0.4306f, 0.0f, 0.6734f,  0.616f, 0.615f, -0.4116f, 0.0f, 0.6845f,  0.619f, 0.621f,
    -0.3918f, 0.0f, 0.6954f,  0.623f, 0.623f, -0.3709f, 0.0f, 0.7059f,  0.626f, 0.626f,
    -0.3486f, 0.0f, 0.7157f,  0.63f,  0.63f,  -0.3251f, 0.0f, 0.7249f,  0.637f, 0.632f,
    -0.3006f, 0.0f, 0.7333f,  0.646f, 0.644f, -0.2755f, 0.0f, 0.7414f,  0.654f, 0.657f,
    -0.25f,   0.0f, 0.7489f,  0.659f, 0.661f, -0.2242f, 0.0f, 0.7562f,  0.664f, 0.664f,
    -0.1979f, 0.0f, 0.7631f,  0.667f, 0.668f, -0.171f,  0.0f, 0.7695f,  0.671f, 0.671f,
    -0.1434f, 0.0f, 0.7752f,  0.674f, 0.674f, -0.1151f, 0.0f, 0.7801f,  0.677f, 0.678f,
    -0.0861f, 0.0f, 0.7841f,  0.678f, 0.68f,  -0.0563f, 0.0f, 0.7869f,  0.68f,  0.68f,
    -0.026f,  0.0f, 0.7889f,  0.68f,  0.68f,  0.0049f,  0.0f, 0.7899f,  0.681f, 0.681f,
    0.0362f,  0.0f, 0.7898f,  0.682f, 0.681f, 0.0679f,  0.0f, 0.7881f,  0.683f, 0.683f,
    0.0996f,  0.0f, 0.7853f,  0.685f, 0.683f, 0.1313f,  0.0f, 0.7812f,  0.687f, 0.685f,
    0.1632f,  0.0f, 0.7756f,  0.69f,  0.686f, 0.1953f,  0.0f, 0.7687f,  0.693f, 0.694f,
    0.2277f,  0.0f, 0.7608f,  0.697f, 0.697f, 0.2606f,  0.0f, 0.7513f,  0.7f,   0.7f,
    0.2934f,  0.0f, 0.7404f,  0.704f, 0.704f, 0.3258f,  0.0f, 0.7276f,  0.707f, 0.71f,
    0.357f,   0.0f, 0.7135f,  0.709f, 0.711f, 0.387f,   0.0f, 0.6983f,  0.711f, 0.713f,
    0.4157f,  0.0f, 0.6819f,  0.712f, 0.714f, 0.444f,   0.0f, 0.6645f,  0.714f, 0.714f,
    0.4719f,  0.0f, 0.6459f,  0.715f, 0.715f, 0.4994f,  0.0f, 0.6261f,  0.715f, 0.716f,
    0.526f,   0.0f, 0.6046f,  0.716f, 0.716f, 0.552f,   0.0f, 0.5816f,  0.717f, 0.717f,
    0.577f,   0.0f, 0.5575f,  0.718f, 0.717f, 0.6008f,  0.0f, 0.5328f,  0.718f, 0.718f,
    0.6231f,  0.0f, 0.5077f,  0.718f, 0.718f, 0.6423f,  0.0f, 0.4829f,  0.719f, 0.718f,
    0.658f,   0.0f, 0.4617f,  0.719f, 0.719f, 0.6713f,  0.0f, 0.4432f,  0.719f, 0.719f,
    0.6828f,  0.0f, 0.4266f,  0.719f, 0.719f, 0.6928f,  0.0f, 0.4118f,  0.719f, 0.719f,
    0.7016f,  0.0f, 0.3987f,  0.718f, 0.717f, 0.7094f,  0.0f, 0.3871f,  0.717f, 0.717f,
    0.7165f,  0.0f, 0.3769f,  0.717f, 0.717f, 0.7233f,  0.0f, 0.3679f,  0.718f, 0.718f,
    0.7301f,  0.0f, 0.3598f,  0.717f, 0.719f, 0.7373f,  0.0f, 0.3524f,  0.715f, 0.719f,
    0.7454f,  0.0f, 0.3458f,  0.713f, 0.709f, 0.7545f,  0.0f, 0.3398f,  0.718f, 0.704f,
    0.7651f,  0.0f, 0.3351f,  0.732f, 0.705f, 0.777f,   0.0f, 0.3317f,  0.753f, 0.713f,
    0.7909f,  0.0f, 0.3311f,  0.774f, 0.813f, 0.8068f,  0.0f, 0.334f,   0.791f, 0.815f,
    0.8246f,  0.0f, 0.3398f,  0.802f, 0.815f, 0.8438f,  0.0f, 0.3486f,  0.809f, 0.816f,
    0.8651f,  0.0f, 0.3575f,  0.812f, 0.816f, 0.8893f,  0.0f, 0.3665f,  0.814f, 0.816f,
    0.9166f,  0.0f, 0.374f,   0.814f, 0.817f, 0.9459f,  0.0f, 0.3791f,  0.812f, 0.817f,
    0.9751f,  0.0f, 0.3811f,  0.81f,  0.815f, 1.0029f,  0.0f, 0.38f,    0.806f, 0.807f,
    1.0288f,  0.0f, 0.3754f,  0.8f,   0.801f, 1.052f,   0.0f, 0.3673f,  0.794f, 0.8f,
    1.0722f,  0.0f, 0.3556f,  0.788f, 0.781f, 1.0888f,  0.0f, 0.3403f,  0.783f, 0.78f,
    1.1027f,  0.0f, 0.322f,   0.781f, 0.778f, 1.1133f,  0.0f, 0.301f,   0.779f, 0.777f,
    1.1215f,  0.0f, 0.278f,   0.778f, 0.777f, 1.1269f,  0.0f, 0.2534f,  0.777f, 0.777f,
    1.1296f,  0.0f, 0.2284f,  0.777f, 0.778f, 1.1292f,  0.0f, 0.2031f,  0.776f, 0.776f,
    1.1254f,  0.0f, 0.1778f,  0.775f, 0.776f, 1.1178f,  0.0f, 0.153f,   0.774f, 0.774f,
    1.1076f,  0.0f, 0.1299f,  0.774f, 0.772f, 1.0955f,  0.0f, 0.1079f,  0.773f, 0.773f,
    1.0817f,  0.0f, 0.087f,   0.772f, 0.773f, 1.0668f,  0.0f, 0.0677f,  0.771f, 0.772f,
    1.0508f,  0.0f, 0.0491f,  0.77f,  0.772f, 1.0339f,  0.0f, 0.0313f,  0.768f, 0.766f,
    1.0157f,  0.0f, 0.0144f,  0.767f, 0.765f, 0.9969f,  0.0f, -0.0015f, 0.766f, 0.765f,
    0.9784f,  0.0f, -0.017f,  0.765f, 0.766f, 0.96f,    0.0f, -0.0321f, 0.764f, 0.765f,
    0.9413f,  0.0f, -0.0468f, 0.761f, 0.765f, 0.9216f,  0.0f, -0.0611f, 0.756f, 0.761f,
    0.9009f,  0.0f, -0.0751f, 0.751f, 0.751f, 0.8787f,  0.0f, -0.0893f, 0.745f, 0.744f,
    0.8556f,  0.0f, -0.1027f, 0.739f, 0.738f, 0.8312f,  0.0f, -0.1152f, 0.733f, 0.731f,
    0.8058f,  0.0f, -0.1268f, 0.728f, 0.727f, 0.7788f,  0.0f, -0.1372f, 0.723f, 0.723f,
    0.7505f,  0.0f, -0.1467f, 0.718f, 0.717f, 0.7214f,  0.0f, -0.1549f, 0.713f, 0.708f,
    0.6929f,  0.0f, -0.1617f, 0.709f, 0.706f, 0.6652f,  0.0f, -0.1665f, 0.704f, 0.705f,
    0.6388f,  0.0f, -0.1691f, 0.7f,   0.704f, 0.6131f,  0.0f, -0.1701f, 0.695f, 0.698f,
    0.5883f,  0.0f, -0.1699f, 0.691f, 0.691f, 0.5644f,  0.0f, -0.1691f, 0.686f, 0.685f,
    0.5416f,  0.0f, -0.1683f, 0.681f, 0.683f, 0.5195f,  0.0f, -0.168f,  0.676f, 0.676f,
    0.4975f,  0.0f, -0.1687f, 0.671f, 0.67f,  0.4754f,  0.0f, -0.1705f, 0.666f, 0.663f,
    0.4527f,  0.0f, -0.1741f, 0.663f, 0.66f,  0.4293f,  0.0f, -0.1797f, 0.661f, 0.659f,
    0.4054f,  0.0f, -0.1881f, 0.66f,  0.659f, 0.3813f,  0.0f, -0.1992f, 0.659f, 0.657f,
    0.3585f,  0.0f, -0.212f,  0.658f, 0.659f, 0.3368f,  0.0f, -0.2266f, 0.658f, 0.659f,
    0.3174f,  0.0f, -0.2426f, 0.658f, 0.659f, 0.2996f,  0.0f, -0.2594f, 0.657f, 0.657f,
    0.284f,   0.0f, -0.2768f, 0.656f, 0.658f, 0.2702f,  0.0f, -0.2946f, 0.653f, 0.657f,
    0.2585f,  0.0f, -0.3127f, 0.646f, 0.656f, 0.25f,    0.0f, -0.3308f, 0.637f, 0.642f,
    0.2447f,  0.0f, -0.3489f, 0.628f, 0.609f, 0.2418f,  0.0f, -0.3672f, 0.62f,  0.608f,
    0.2412f,  0.0f, -0.386f,  0.614f, 0.607f, 0.2425f,  0.0f, -0.4051f, 0.61f,  0.606f,
    0.2456f,  0.0f, -0.4246f, 0.608f, 0.604f, 0.2509f,  0.0f, -0.4447f, 0.607f, 0.606f,
    0.2576f,  0.0f, -0.4652f, 0.606f, 0.607f, 0.2666f,  0.0f, -0.4867f, 0.605f, 0.607f,
    0.2766f,  0.0f, -0.5091f, 0.603f, 0.607f, 0.2871f,  0.0f, -0.5326f, 0.598f, 0.606f,
    0.2973f,  0.0f, -0.5569f, 0.591f, 0.602f, 0.306f,   0.0f, -0.5826f, 0.583f, 0.585f,
    0.3131f,  0.0f, -0.61f,   0.574f, 0.576f, 0.3197f,  0.0f, -0.6384f, 0.564f, 0.564f,
    0.326f,   0.0f, -0.6681f, 0.555f, 0.549f, 0.3315f,  0.0f, -0.6984f, 0.547f, 0.543f,
    0.336f,   0.0f, -0.7291f, 0.541f, 0.541f, 0.3391f,  0.0f, -0.7593f, 0.536f, 0.538f,
    0.3399f,  0.0f, -0.7884f, 0.532f, 0.528f, 0.3382f,  0.0f, -0.8158f, 0.529f, 0.528f,
    0.334f,   0.0f, -0.8417f, 0.525f, 0.529f, 0.3273f,  0.0f, -0.8657f, 0.521f, 0.528f,
    0.3185f,  0.0f, -0.8881f, 0.516f, 0.515f, 0.3073f,  0.0f, -0.9088f, 0.51f,  0.514f,
    0.2941f,  0.0f, -0.9278f, 0.505f, 0.507f, 0.2786f,  0.0f, -0.9449f, 0.499f, 0.494f,
    0.261f,   0.0f, -0.96f,   0.495f, 0.49f,  0.2413f,  0.0f, -0.9733f, 0.493f, 0.491f,
    0.2193f,  0.0f, -0.9845f, 0.491f, 0.489f, 0.1953f,  0.0f, -0.9935f, 0.491f, 0.491f,
    0.1693f,  0.0f, -1.0004f, 0.491f, 0.492f, 0.1421f,  0.0f, -1.0051f, 0.492f, 0.492f,
    0.1136f,  0.0f, -1.0072f, 0.492f, 0.492f, 0.0842f,  0.0f, -1.0073f, 0.492f, 0.492f,
    0.0548f,  0.0f, -1.0059f, 0.493f, 0.494f, 0.0258f,  0.0f, -1.0037f, 0.493f, 0.494f,
    -0.0027f, 0.0f, -1.0003f, 0.493f, 0.493f, -0.0309f, 0.0f, -0.9959f, 0.492f, 0.492f,
    -0.0584f, 0.0f, -0.9904f, 0.491f, 0.492f, -0.0858f, 0.0f, -0.9848f, 0.491f, 0.491f,
    -0.1127f, 0.0f, -0.9783f, 0.49f,  0.49f,  -0.1386f, 0.0f, -0.9703f, 0.49f,  0.49f,
    -0.1649f, 0.0f, -0.9604f, 0.489f, 0.489f, -0.191f,  0.0f, -0.9479f, 0.489f, 0.489f,
    -0.2165f, 0.0f, -0.9345f, 0.489f, 0.49f,  -0.2414f, 0.0f, -0.9205f, 0.489f, 0.489f,
    -0.2654f, 0.0f, -0.9055f, 0.489f, 0.489f, -0.2877f, 0.0f, -0.8898f, 0.49f,  0.49f,
    -0.3076f, 0.0f, -0.8723f, 0.49f,  0.489f, -0.324f,  0.0f, -0.8532f, 0.491f, 0.489f,
    -0.3367f, 0.0f, -0.8316f, 0.492f, 0.489f, -0.3451f, 0.0f, -0.8077f, 0.494f, 0.488f,
    -0.3505f, 0.0f, -0.7829f, 0.497f, 0.49f,  -0.3531f, 0.0f, -0.7584f, 0.501f, 0.497f,
    -0.3528f, 0.0f, -0.7349f, 0.505f, 0.504f, -0.3503f, 0.0f, -0.7115f, 0.51f,  0.51f,
    -0.346f,  0.0f, -0.688f,  0.515f, 0.515f, -0.3411f, 0.0f, -0.6643f, 0.52f,  0.522f,
    -0.3361f, 0.0f, -0.6403f, 0.525f, 0.528f, -0.3304f, 0.0f, -0.6164f, 0.53f,  0.532f,
    -0.3244f, 0.0f, -0.5925f, 0.535f, 0.535f, -0.318f,  0.0f, -0.5687f, 0.539f, 0.54f,
    -0.3124f, 0.0f, -0.5441f, 0.542f, 0.545f, -0.3051f, 0.0f, -0.5191f, 0.546f, 0.549f,
    -0.2959f, 0.0f, -0.4917f, 0.548f, 0.549f, -0.2882f, 0.0f, -0.4639f, 0.55f,  0.552f,
    -0.2814f, 0.0f, -0.4363f, 0.551f, 0.552f, -0.2759f, 0.0f, -0.4084f, 0.552f, 0.553f,
    -0.2707f, 0.0f, -0.3827f, 0.553f, 0.553f, -0.2703f, 0.0f, -0.3586f, 0.554f, 0.553f,
    -0.2772f, 0.0f, -0.3375f, 0.554f, 0.554f, -0.2871f, 0.0f, -0.3178f, 0.555f, 0.555f,
    -0.2995f, 0.0f, -0.2996f, 0.556f, 0.555f, -0.3145f, 0.0f, -0.283f,  0.556f, 0.557f,
    -0.332f,  0.0f, -0.2672f, 0.557f, 0.557f, -0.3488f, 0.0f, -0.2531f, 0.558f, 0.558f,
    -0.3639f, 0.0f, -0.2407f, 0.558f, 0.558f, -0.3778f, 0.0f, -0.2292f, 0.558f, 0.558f,
    -0.3909f, 0.0f, -0.2191f, 0.559f, 0.559f, -0.4032f, 0.0f, -0.2102f, 0.559f, 0.558f,
    -0.4146f, 0.0f, -0.2027f, 0.559f, 0.559f, -0.426f,  0.0f, -0.1968f, 0.558f, 0.558f,
    -0.4348f, 0.0f, -0.1931f, 0.558f, 0.558f, -0.4479f, 0.0f, -0.1886f, 0.555f, 0.559f,
};

static const float data1[33 * GP_PRIM_DATABUF_SIZE] = {
    0.5292f, 0.0f, 0.1742f,  0.1f,    1.0f, 0.5291f, 0.0f, 0.1621f,  0.2199f, 1.0f,
    0.5274f, 0.0f, 0.1386f,  0.4615f, 1.0f, 0.5239f, 0.0f, 0.116f,   0.6019f, 1.0f,
    0.5185f, 0.0f, 0.0945f,  0.6981f, 1.0f, 0.5115f, 0.0f, 0.0741f,  0.7689f, 1.0f,
    0.503f,  0.0f, 0.0548f,  0.8236f, 1.0f, 0.4931f, 0.0f, 0.0368f,  0.866f,  1.0f,
    0.482f,  0.0f, 0.02f,    0.9003f, 1.0f, 0.4697f, 0.0f, 0.0046f,  0.9272f, 1.0f,
    0.4565f, 0.0f, -0.0094f, 0.9485f, 1.0f, 0.4424f, 0.0f, -0.0219f, 0.9653f, 1.0f,
    0.4275f, 0.0f, -0.033f,  0.9781f, 1.0f, 0.4121f, 0.0f, -0.0424f, 0.9876f, 1.0f,
    0.3961f, 0.0f, -0.0501f, 0.9942f, 1.0f, 0.3799f, 0.0f, -0.0562f, 0.9983f, 1.0f,
    0.3634f, 0.0f, -0.0605f, 0.9997f, 1.0f, 0.3468f, 0.0f, -0.0629f, 0.999f,  1.0f,
    0.3303f, 0.0f, -0.0634f, 0.9963f, 1.0f, 0.3139f, 0.0f, -0.062f,  0.9912f, 1.0f,
    0.2979f, 0.0f, -0.0585f, 0.9834f, 1.0f, 0.2823f, 0.0f, -0.0529f, 0.9724f, 1.0f,
    0.2672f, 0.0f, -0.0452f, 0.9576f, 1.0f, 0.2529f, 0.0f, -0.0352f, 0.9385f, 1.0f,
    0.2393f, 0.0f, -0.023f,  0.9143f, 1.0f, 0.2267f, 0.0f, -0.0085f, 0.8841f, 1.0f,
    0.2153f, 0.0f, 0.0085f,  0.8461f, 1.0f, 0.205f,  0.0f, 0.0279f,  0.7979f, 1.0f,
    0.196f,  0.0f, 0.0499f,  0.7359f, 1.0f, 0.1886f, 0.0f, 0.0745f,  0.6541f, 1.0f,
    0.1827f, 0.0f, 0.1017f,  0.5396f, 1.0f, 0.1786f, 0.0f, 0.1316f,  0.36f,   1.0f,
    0.1763f, 0.0f, 0.1643f,  0.1f,    1.0f,
};

static const float data2[18 * GP_PRIM_DATABUF_SIZE] = {
    -0.0844f, 0.0f, -0.301f,  0.1f,    1.0f, -0.0825f, 0.0f, -0.3034f, 0.2199f, 1.0f,
    -0.0751f, 0.0f, -0.3128f, 0.6019f, 1.0f, -0.0677f, 0.0f, -0.3216f, 0.7689f, 1.0f,
    -0.06f,   0.0f, -0.3298f, 0.866f,  1.0f, -0.0522f, 0.0f, -0.3372f, 0.9272f, 1.0f,
    -0.044f,  0.0f, -0.3437f, 0.9653f, 1.0f, -0.0354f, 0.0f, -0.3491f, 0.9876f, 1.0f,
    -0.0264f, 0.0f, -0.3535f, 0.9983f, 1.0f, -0.0168f, 0.0f, -0.3566f, 0.999f,  1.0f,
    -0.0065f, 0.0f, -0.3583f, 0.9912f, 1.0f, 0.0045f,  0.0f, -0.3585f, 0.9724f, 1.0f,
    0.0163f,  0.0f, -0.3571f, 0.9385f, 1.0f, 0.029f,   0.0f, -0.354f,  0.8841f, 1.0f,
    0.0427f,  0.0f, -0.3491f, 0.7979f, 1.0f, 0.0574f,  0.0f, -0.3421f, 0.6541f, 1.0f,
    0.0732f,  0.0f, -0.3331f, 0.36f,   1.0f, 0.0816f,  0.0f, -0.3278f, 0.1f,    1.0f,
};

static const float data3[64 * GP_PRIM_DATABUF_SIZE] = {
    -0.6551f, 0.0f, 0.4448f, 0.1f,    1.0f, -0.6353f, 0.0f, 0.4689f, 0.2199f, 1.0f,
    -0.6211f, 0.0f, 0.4845f, 0.36f,   1.0f, -0.6033f, 0.0f, 0.5034f, 0.4615f, 1.0f,
    -0.5856f, 0.0f, 0.5211f, 0.5396f, 1.0f, -0.5672f, 0.0f, 0.5387f, 0.6019f, 1.0f,
    -0.5485f, 0.0f, 0.5555f, 0.6541f, 1.0f, -0.5295f, 0.0f, 0.5718f, 0.6981f, 1.0f,
    -0.5103f, 0.0f, 0.5875f, 0.7359f, 1.0f, -0.4909f, 0.0f, 0.6028f, 0.7689f, 1.0f,
    -0.4712f, 0.0f, 0.6174f, 0.7979f, 1.0f, -0.4512f, 0.0f, 0.6313f, 0.8236f, 1.0f,
    -0.4307f, 0.0f, 0.6444f, 0.8461f, 1.0f, -0.4099f, 0.0f, 0.6568f, 0.866f,  1.0f,
    -0.3884f, 0.0f, 0.6684f, 0.8841f, 1.0f, -0.3665f, 0.0f, 0.6793f, 0.9003f, 1.0f,
    -0.3439f, 0.0f, 0.6893f, 0.9143f, 1.0f, -0.3207f, 0.0f, 0.6984f, 0.9272f, 1.0f,
    -0.2971f, 0.0f, 0.7069f, 0.9385f, 1.0f, -0.2731f, 0.0f, 0.7148f, 0.9485f, 1.0f,
    -0.249f,  0.0f, 0.7222f, 0.9576f, 1.0f, -0.2247f, 0.0f, 0.7292f, 0.9653f, 1.0f,
    -0.2003f, 0.0f, 0.7356f, 0.9724f, 1.0f, -0.1759f, 0.0f, 0.7416f, 0.9781f, 1.0f,
    -0.1515f, 0.0f, 0.7471f, 0.9834f, 1.0f, -0.1272f, 0.0f, 0.7518f, 0.9876f, 1.0f,
    -0.1028f, 0.0f, 0.7556f, 0.9912f, 1.0f, -0.0785f, 0.0f, 0.7586f, 0.9942f, 1.0f,
    -0.0543f, 0.0f, 0.7607f, 0.9963f, 1.0f, -0.0302f, 0.0f, 0.7621f, 0.9983f, 1.0f,
    -0.0062f, 0.0f, 0.7627f, 0.999f,  1.0f, 0.0177f,  0.0f, 0.7625f, 0.9997f, 1.0f,
    0.0415f,  0.0f, 0.7616f, 0.9997f, 1.0f, 0.0652f,  0.0f, 0.7602f, 0.999f,  1.0f,
    0.0887f,  0.0f, 0.7583f, 0.9983f, 1.0f, 0.1122f,  0.0f, 0.7559f, 0.9963f, 1.0f,
    0.1355f,  0.0f, 0.7529f, 0.9942f, 1.0f, 0.1585f,  0.0f, 0.7493f, 0.9912f, 1.0f,
    0.1814f,  0.0f, 0.7451f, 0.9876f, 1.0f, 0.2041f,  0.0f, 0.7404f, 0.9834f, 1.0f,
    0.2266f,  0.0f, 0.7352f, 0.9781f, 1.0f, 0.2488f,  0.0f, 0.729f,  0.9724f, 1.0f,
    0.2706f,  0.0f, 0.7216f, 0.9653f, 1.0f, 0.2921f,  0.0f, 0.7131f, 0.9576f, 1.0f,
    0.3135f,  0.0f, 0.7041f, 0.9485f, 1.0f, 0.3348f,  0.0f, 0.6945f, 0.9385f, 1.0f,
    0.3559f,  0.0f, 0.6845f, 0.9272f, 1.0f, 0.3769f,  0.0f, 0.6739f, 0.9143f, 1.0f,
    0.3978f,  0.0f, 0.6628f, 0.9003f, 1.0f, 0.4185f,  0.0f, 0.651f,  0.8841f, 1.0f,
    0.439f,   0.0f, 0.6383f, 0.866f,  1.0f, 0.4594f,  0.0f, 0.6249f, 0.8461f, 1.0f,
    0.4795f,  0.0f, 0.6106f, 0.8236f, 1.0f, 0.4995f,  0.0f, 0.5956f, 0.7979f, 1.0f,
    0.5193f,  0.0f, 0.5801f, 0.7689f, 1.0f, 0.539f,   0.0f, 0.5642f, 0.7359f, 1.0f,
    0.5586f,  0.0f, 0.5479f, 0.6981f, 1.0f, 0.578f,   0.0f, 0.5312f, 0.6541f, 1.0f,
    0.597f,   0.0f, 0.5141f, 0.6019f, 1.0f, 0.6153f,  0.0f, 0.4966f, 0.5396f, 1.0f,
    0.6324f,  0.0f, 0.4797f, 0.4615f, 1.0f, 0.6498f,  0.0f, 0.462f,  0.36f,   1.0f,
    0.6638f,  0.0f, 0.4477f, 0.2199f, 1.0f, 0.6843f,  0.0f, 0.4263f, 0.1f,    1.0f,
};

static const float data4[33 * GP_PRIM_DATABUF_SIZE] = {
    -0.7765f, 0.0f, 0.3255f, 0.1f,    1.0f, -0.7908f, 0.0f, 0.3335f, 0.2199f, 1.0f,
    -0.8048f, 0.0f, 0.3402f, 0.4615f, 1.0f, -0.8225f, 0.0f, 0.3467f, 0.6019f, 1.0f,
    -0.8426f, 0.0f, 0.3529f, 0.6981f, 1.0f, -0.8651f, 0.0f, 0.3587f, 0.7689f, 1.0f,
    -0.8878f, 0.0f, 0.3633f, 0.8236f, 1.0f, -0.9105f, 0.0f, 0.367f,  0.866f,  1.0f,
    -0.9323f, 0.0f, 0.3693f, 0.9003f, 1.0f, -0.9529f, 0.0f, 0.3701f, 0.9272f, 1.0f,
    -0.972f,  0.0f, 0.3695f, 0.9485f, 1.0f, -0.9896f, 0.0f, 0.3675f, 0.9653f, 1.0f,
    -1.0057f, 0.0f, 0.364f,  0.9781f, 1.0f, -1.0203f, 0.0f, 0.3592f, 0.9876f, 1.0f,
    -1.0333f, 0.0f, 0.3528f, 0.9942f, 1.0f, -1.0447f, 0.0f, 0.3452f, 0.9983f, 1.0f,
    -1.0546f, 0.0f, 0.3362f, 0.9997f, 1.0f, -1.0633f, 0.0f, 0.3261f, 0.999f,  1.0f,
    -1.0708f, 0.0f, 0.315f,  0.9963f, 1.0f, -1.0767f, 0.0f, 0.3027f, 0.9912f, 1.0f,
    -1.0818f, 0.0f, 0.2894f, 0.9834f, 1.0f, -1.0861f, 0.0f, 0.2752f, 0.9724f, 1.0f,
    -1.0897f, 0.0f, 0.2604f, 0.9576f, 1.0f, -1.0922f, 0.0f, 0.2446f, 0.9385f, 1.0f,
    -1.0938f, 0.0f, 0.2277f, 0.9143f, 1.0f, -1.0944f, 0.0f, 0.2098f, 0.8841f, 1.0f,
    -1.0939f, 0.0f, 0.191f,  0.8461f, 1.0f, -1.0924f, 0.0f, 0.1714f, 0.7979f, 1.0f,
    -1.0897f, 0.0f, 0.1511f, 0.7359f, 1.0f, -1.0855f, 0.0f, 0.1303f, 0.6541f, 1.0f,
    -1.0798f, 0.0f, 0.1095f, 0.5396f, 1.0f, -1.0723f, 0.0f, 0.089f,  0.36f,   1.0f,
    -1.0642f, 0.0f, 0.0702f, 0.1f,    1.0f,
};

static const float data5[64 * GP_PRIM_DATABUF_SIZE] = {
    0.8135f, 0.0f, 0.3341f, 0.1f,    1.0f, 0.8191f, 0.0f, 0.3376f, 0.2199f, 1.0f,
    0.8246f, 0.0f, 0.3408f, 0.36f,   1.0f, 0.8304f, 0.0f, 0.3438f, 0.4615f, 1.0f,
    0.8368f, 0.0f, 0.3465f, 0.5396f, 1.0f, 0.843f,  0.0f, 0.3491f, 0.6019f, 1.0f,
    0.8494f, 0.0f, 0.3515f, 0.6541f, 1.0f, 0.8558f, 0.0f, 0.3536f, 0.6981f, 1.0f,
    0.8623f, 0.0f, 0.3557f, 0.7359f, 1.0f, 0.8688f, 0.0f, 0.3575f, 0.7689f, 1.0f,
    0.8752f, 0.0f, 0.3593f, 0.7979f, 1.0f, 0.8813f, 0.0f, 0.3609f, 0.8236f, 1.0f,
    0.8872f, 0.0f, 0.3625f, 0.8461f, 1.0f, 0.8929f, 0.0f, 0.364f,  0.866f,  1.0f,
    0.8984f, 0.0f, 0.3653f, 0.8841f, 1.0f, 0.9039f, 0.0f, 0.3665f, 0.9003f, 1.0f,
    0.9093f, 0.0f, 0.3675f, 0.9143f, 1.0f, 0.9146f, 0.0f, 0.3684f, 0.9272f, 1.0f,
    0.9199f, 0.0f, 0.3692f, 0.9385f, 1.0f, 0.9253f, 0.0f, 0.3698f, 0.9485f, 1.0f,
    0.9305f, 0.0f, 0.3703f, 0.9576f, 1.0f, 0.9358f, 0.0f, 0.3706f, 0.9653f, 1.0f,
    0.941f,  0.0f, 0.3709f, 0.9724f, 1.0f, 0.9462f, 0.0f, 0.371f,  0.9781f, 1.0f,
    0.9514f, 0.0f, 0.371f,  0.9834f, 1.0f, 0.9566f, 0.0f, 0.3708f, 0.9876f, 1.0f,
    0.9617f, 0.0f, 0.3706f, 0.9912f, 1.0f, 0.9668f, 0.0f, 0.3702f, 0.9942f, 1.0f,
    0.9718f, 0.0f, 0.3697f, 0.9963f, 1.0f, 0.9768f, 0.0f, 0.3692f, 0.9983f, 1.0f,
    0.9818f, 0.0f, 0.3685f, 0.999f,  1.0f, 0.9867f, 0.0f, 0.3677f, 0.9997f, 1.0f,
    0.9916f, 0.0f, 0.3667f, 0.9997f, 1.0f, 0.9964f, 0.0f, 0.3657f, 0.999f,  1.0f,
    1.0012f, 0.0f, 0.3646f, 0.9983f, 1.0f, 1.006f,  0.0f, 0.3634f, 0.9963f, 1.0f,
    1.0107f, 0.0f, 0.3621f, 0.9942f, 1.0f, 1.0154f, 0.0f, 0.3607f, 0.9912f, 1.0f,
    1.02f,   0.0f, 0.3593f, 0.9876f, 1.0f, 1.0245f, 0.0f, 0.3577f, 0.9834f, 1.0f,
    1.029f,  0.0f, 0.3561f, 0.9781f, 1.0f, 1.0335f, 0.0f, 0.3543f, 0.9724f, 1.0f,
    1.0379f, 0.0f, 0.3525f, 0.9653f, 1.0f, 1.0422f, 0.0f, 0.3507f, 0.9576f, 1.0f,
    1.0465f, 0.0f, 0.3487f, 0.9485f, 1.0f, 1.0507f, 0.0f, 0.3468f, 0.9385f, 1.0f,
    1.0549f, 0.0f, 0.3447f, 0.9272f, 1.0f, 1.0591f, 0.0f, 0.3427f, 0.9143f, 1.0f,
    1.0633f, 0.0f, 0.3404f, 0.9003f, 1.0f, 1.0675f, 0.0f, 0.338f,  0.8841f, 1.0f,
    1.0717f, 0.0f, 0.3351f, 0.866f,  1.0f, 1.0761f, 0.0f, 0.3318f, 0.8461f, 1.0f,
    1.0805f, 0.0f, 0.3279f, 0.8236f, 1.0f, 1.0849f, 0.0f, 0.3235f, 0.7979f, 1.0f,
    1.0893f, 0.0f, 0.3186f, 0.7689f, 1.0f, 1.0936f, 0.0f, 0.3134f, 0.7359f, 1.0f,
    1.0979f, 0.0f, 0.3076f, 0.6981f, 1.0f, 1.102f,  0.0f, 0.3017f, 0.6541f, 1.0f,
    1.106f,  0.0f, 0.2953f, 0.6019f, 1.0f, 1.1097f, 0.0f, 0.289f,  0.5396f, 1.0f,
    1.1132f, 0.0f, 0.2826f, 0.4615f, 1.0f, 1.1164f, 0.0f, 0.2766f, 0.36f,   1.0f,
    1.1193f, 0.0f, 0.2708f, 0.2199f, 1.0f, 1.1221f, 0.0f, 0.2652f, 0.1f,    1.0f,
};

static const float data6[33 * GP_PRIM_DATABUF_SIZE] = {
    -0.2677f, 0.0f, -0.3496f, 0.1f,    1.0f, -0.2658f, 0.0f, -0.3919f, 0.2199f, 1.0f,
    -0.2657f, 0.0f, -0.4295f, 0.4615f, 1.0f, -0.2691f, 0.0f, -0.4621f, 0.6019f, 1.0f,
    -0.275f,  0.0f, -0.4949f, 0.6981f, 1.0f, -0.2828f, 0.0f, -0.527f,  0.7689f, 1.0f,
    -0.2911f, 0.0f, -0.5578f, 0.8236f, 1.0f, -0.2971f, 0.0f, -0.5884f, 0.866f,  1.0f,
    -0.303f,  0.0f, -0.6181f, 0.9003f, 1.0f, -0.3089f, 0.0f, -0.6467f, 0.9272f, 1.0f,
    -0.3148f, 0.0f, -0.6738f, 0.9485f, 1.0f, -0.3196f, 0.0f, -0.7f,    0.9653f, 1.0f,
    -0.323f,  0.0f, -0.7253f, 0.9781f, 1.0f, -0.3234f, 0.0f, -0.7496f, 0.9876f, 1.0f,
    -0.3219f, 0.0f, -0.7728f, 0.9942f, 1.0f, -0.3184f, 0.0f, -0.7949f, 0.9983f, 1.0f,
    -0.3103f, 0.0f, -0.8138f, 0.9997f, 1.0f, -0.3004f, 0.0f, -0.8307f, 0.999f,  1.0f,
    -0.2892f, 0.0f, -0.8464f, 0.9963f, 1.0f, -0.2766f, 0.0f, -0.8611f, 0.9912f, 1.0f,
    -0.2618f, 0.0f, -0.8735f, 0.9834f, 1.0f, -0.2469f, 0.0f, -0.8853f, 0.9724f, 1.0f,
    -0.2315f, 0.0f, -0.8964f, 0.9576f, 1.0f, -0.2158f, 0.0f, -0.9068f, 0.9385f, 1.0f,
    -0.1997f, 0.0f, -0.9167f, 0.9143f, 1.0f, -0.1833f, 0.0f, -0.9262f, 0.8841f, 1.0f,
    -0.1663f, 0.0f, -0.9355f, 0.8461f, 1.0f, -0.1481f, 0.0f, -0.9445f, 0.7979f, 1.0f,
    -0.1282f, 0.0f, -0.9534f, 0.7359f, 1.0f, -0.1077f, 0.0f, -0.9625f, 0.6541f, 1.0f,
    -0.0854f, 0.0f, -0.9718f, 0.5396f, 1.0f, -0.0616f, 0.0f, -0.9813f, 0.36f,   1.0f,
    -0.0248f, 0.0f, -0.992f,  0.1f,    1.0f,
};

static const float data7[18 * GP_PRIM_DATABUF_SIZE] = {
    -0.0618f, 0.0f, -0.1922f, 0.1f,    1.0f, -0.0703f, 0.0f, -0.2021f, 0.2199f, 1.0f,
    -0.0758f, 0.0f, -0.2103f, 0.6019f, 1.0f, -0.0803f, 0.0f, -0.2206f, 0.7689f, 1.0f,
    -0.083f,  0.0f, -0.2307f, 0.866f,  1.0f, -0.0851f, 0.0f, -0.2405f, 0.9272f, 1.0f,
    -0.0865f, 0.0f, -0.2499f, 0.9653f, 1.0f, -0.0872f, 0.0f, -0.2588f, 0.9876f, 1.0f,
    -0.0869f, 0.0f, -0.2673f, 0.9983f, 1.0f, -0.0858f, 0.0f, -0.2754f, 0.999f,  1.0f,
    -0.0831f, 0.0f, -0.2829f, 0.9912f, 1.0f, -0.0791f, 0.0f, -0.2898f, 0.9724f, 1.0f,
    -0.074f,  0.0f, -0.2966f, 0.9385f, 1.0f, -0.0674f, 0.0f, -0.303f,  0.8841f, 1.0f,
    -0.0591f, 0.0f, -0.3084f, 0.7979f, 1.0f, -0.0465f, 0.0f, -0.3134f, 0.6541f, 1.0f,
    -0.0331f, 0.0f, -0.3165f, 0.36f,   1.0f, -0.015f,  0.0f, -0.318f,  0.1f,    1.0f,
};

static const float data8[49 * GP_PRIM_DATABUF_SIZE] = {
    0.5311f, 0.0f, 0.1661f,  1.0f, 1.0f, 0.5307f, 0.0f, 0.1794f,  1.0f, 1.0f,
    0.5277f, 0.0f, 0.2057f,  1.0f, 1.0f, 0.5218f, 0.0f, 0.2314f,  1.0f, 1.0f,
    0.513f,  0.0f, 0.256f,   1.0f, 1.0f, 0.5014f, 0.0f, 0.279f,   1.0f, 1.0f,
    0.4874f, 0.0f, 0.3001f,  1.0f, 1.0f, 0.4711f, 0.0f, 0.3189f,  1.0f, 1.0f,
    0.4529f, 0.0f, 0.3351f,  1.0f, 1.0f, 0.4329f, 0.0f, 0.3483f,  1.0f, 1.0f,
    0.4117f, 0.0f, 0.3585f,  1.0f, 1.0f, 0.3894f, 0.0f, 0.3654f,  1.0f, 1.0f,
    0.3666f, 0.0f, 0.3689f,  1.0f, 1.0f, 0.3435f, 0.0f, 0.3689f,  1.0f, 1.0f,
    0.3207f, 0.0f, 0.3654f,  1.0f, 1.0f, 0.2985f, 0.0f, 0.3585f,  1.0f, 1.0f,
    0.2772f, 0.0f, 0.3483f,  1.0f, 1.0f, 0.2573f, 0.0f, 0.3351f,  1.0f, 1.0f,
    0.239f,  0.0f, 0.3189f,  1.0f, 1.0f, 0.2227f, 0.0f, 0.3001f,  1.0f, 1.0f,
    0.2087f, 0.0f, 0.279f,   1.0f, 1.0f, 0.1972f, 0.0f, 0.256f,   1.0f, 1.0f,
    0.1884f, 0.0f, 0.2314f,  1.0f, 1.0f, 0.1824f, 0.0f, 0.2057f,  1.0f, 1.0f,
    0.1794f, 0.0f, 0.1794f,  1.0f, 1.0f, 0.1794f, 0.0f, 0.1528f,  1.0f, 1.0f,
    0.1824f, 0.0f, 0.1264f,  1.0f, 1.0f, 0.1884f, 0.0f, 0.1007f,  1.0f, 1.0f,
    0.1972f, 0.0f, 0.0762f,  1.0f, 1.0f, 0.2087f, 0.0f, 0.0531f,  1.0f, 1.0f,
    0.2227f, 0.0f, 0.032f,   1.0f, 1.0f, 0.239f,  0.0f, 0.0132f,  1.0f, 1.0f,
    0.2573f, 0.0f, -0.0029f, 1.0f, 1.0f, 0.2772f, 0.0f, -0.0162f, 1.0f, 1.0f,
    0.2985f, 0.0f, -0.0264f, 1.0f, 1.0f, 0.3207f, 0.0f, -0.0333f, 1.0f, 1.0f,
    0.3435f, 0.0f, -0.0368f, 1.0f, 1.0f, 0.3666f, 0.0f, -0.0368f, 1.0f, 1.0f,
    0.3894f, 0.0f, -0.0333f, 1.0f, 1.0f, 0.4117f, 0.0f, -0.0264f, 1.0f, 1.0f,
    0.4329f, 0.0f, -0.0162f, 1.0f, 1.0f, 0.4529f, 0.0f, -0.0029f, 1.0f, 1.0f,
    0.4711f, 0.0f, 0.0132f,  1.0f, 1.0f, 0.4874f, 0.0f, 0.032f,   1.0f, 1.0f,
    0.5014f, 0.0f, 0.0531f,  1.0f, 1.0f, 0.513f,  0.0f, 0.0762f,  1.0f, 1.0f,
    0.5218f, 0.0f, 0.1007f,  1.0f, 1.0f, 0.5277f, 0.0f, 0.1264f,  1.0f, 1.0f,
    0.5307f, 0.0f, 0.1528f,  1.0f, 1.0f,
};

static const float data9[33 * GP_PRIM_DATABUF_SIZE] = {
    -0.5271f, 0.0f, 0.1742f,  0.1f,    1.0f, -0.527f,  0.0f, 0.1621f,  0.2199f, 1.0f,
    -0.5253f, 0.0f, 0.1386f,  0.4615f, 1.0f, -0.5217f, 0.0f, 0.116f,   0.6019f, 1.0f,
    -0.5164f, 0.0f, 0.0945f,  0.6981f, 1.0f, -0.5094f, 0.0f, 0.0741f,  0.7689f, 1.0f,
    -0.5009f, 0.0f, 0.0548f,  0.8236f, 1.0f, -0.491f,  0.0f, 0.0368f,  0.866f,  1.0f,
    -0.4799f, 0.0f, 0.02f,    0.9003f, 1.0f, -0.4676f, 0.0f, 0.0046f,  0.9272f, 1.0f,
    -0.4544f, 0.0f, -0.0094f, 0.9485f, 1.0f, -0.4402f, 0.0f, -0.0219f, 0.9653f, 1.0f,
    -0.4254f, 0.0f, -0.033f,  0.9781f, 1.0f, -0.4099f, 0.0f, -0.0424f, 0.9876f, 1.0f,
    -0.394f,  0.0f, -0.0501f, 0.9942f, 1.0f, -0.3777f, 0.0f, -0.0562f, 0.9983f, 1.0f,
    -0.3612f, 0.0f, -0.0605f, 0.9997f, 1.0f, -0.3447f, 0.0f, -0.0629f, 0.999f,  1.0f,
    -0.3281f, 0.0f, -0.0634f, 0.9963f, 1.0f, -0.3118f, 0.0f, -0.062f,  0.9912f, 1.0f,
    -0.2957f, 0.0f, -0.0585f, 0.9834f, 1.0f, -0.2801f, 0.0f, -0.0529f, 0.9724f, 1.0f,
    -0.2651f, 0.0f, -0.0452f, 0.9576f, 1.0f, -0.2507f, 0.0f, -0.0352f, 0.9385f, 1.0f,
    -0.2372f, 0.0f, -0.023f,  0.9143f, 1.0f, -0.2246f, 0.0f, -0.0085f, 0.8841f, 1.0f,
    -0.2131f, 0.0f, 0.0085f,  0.8461f, 1.0f, -0.2028f, 0.0f, 0.0279f,  0.7979f, 1.0f,
    -0.1939f, 0.0f, 0.0499f,  0.7359f, 1.0f, -0.1864f, 0.0f, 0.0745f,  0.6541f, 1.0f,
    -0.1806f, 0.0f, 0.1017f,  0.5396f, 1.0f, -0.1765f, 0.0f, 0.1316f,  0.36f,   1.0f,
    -0.1742f, 0.0f, 0.1643f,  0.1f,    1.0f,
};

static const float data10[49 * GP_PRIM_DATABUF_SIZE] = {
    -0.174f,  0.0f, 0.1661f,  1.0f, 1.0f, -0.1744f, 0.0f, 0.1794f,  1.0f, 1.0f,
    -0.1774f, 0.0f, 0.2057f,  1.0f, 1.0f, -0.1833f, 0.0f, 0.2314f,  1.0f, 1.0f,
    -0.1922f, 0.0f, 0.256f,   1.0f, 1.0f, -0.2037f, 0.0f, 0.279f,   1.0f, 1.0f,
    -0.2177f, 0.0f, 0.3001f,  1.0f, 1.0f, -0.234f,  0.0f, 0.3189f,  1.0f, 1.0f,
    -0.2522f, 0.0f, 0.3351f,  1.0f, 1.0f, -0.2722f, 0.0f, 0.3483f,  1.0f, 1.0f,
    -0.2935f, 0.0f, 0.3585f,  1.0f, 1.0f, -0.3157f, 0.0f, 0.3654f,  1.0f, 1.0f,
    -0.3385f, 0.0f, 0.3689f,  1.0f, 1.0f, -0.3616f, 0.0f, 0.3689f,  1.0f, 1.0f,
    -0.3844f, 0.0f, 0.3654f,  1.0f, 1.0f, -0.4066f, 0.0f, 0.3585f,  1.0f, 1.0f,
    -0.4279f, 0.0f, 0.3483f,  1.0f, 1.0f, -0.4479f, 0.0f, 0.3351f,  1.0f, 1.0f,
    -0.4661f, 0.0f, 0.3189f,  1.0f, 1.0f, -0.4824f, 0.0f, 0.3001f,  1.0f, 1.0f,
    -0.4964f, 0.0f, 0.279f,   1.0f, 1.0f, -0.508f,  0.0f, 0.256f,   1.0f, 1.0f,
    -0.5168f, 0.0f, 0.2314f,  1.0f, 1.0f, -0.5227f, 0.0f, 0.2057f,  1.0f, 1.0f,
    -0.5257f, 0.0f, 0.1794f,  1.0f, 1.0f, -0.5257f, 0.0f, 0.1528f,  1.0f, 1.0f,
    -0.5227f, 0.0f, 0.1264f,  1.0f, 1.0f, -0.5168f, 0.0f, 0.1007f,  1.0f, 1.0f,
    -0.508f,  0.0f, 0.0762f,  1.0f, 1.0f, -0.4964f, 0.0f, 0.0531f,  1.0f, 1.0f,
    -0.4824f, 0.0f, 0.032f,   1.0f, 1.0f, -0.4661f, 0.0f, 0.0132f,  1.0f, 1.0f,
    -0.4479f, 0.0f, -0.0029f, 1.0f, 1.0f, -0.4279f, 0.0f, -0.0162f, 1.0f, 1.0f,
    -0.4066f, 0.0f, -0.0264f, 1.0f, 1.0f, -0.3844f, 0.0f, -0.0333f, 1.0f, 1.0f,
    -0.3616f, 0.0f, -0.0368f, 1.0f, 1.0f, -0.3385f, 0.0f, -0.0368f, 1.0f, 1.0f,
    -0.3157f, 0.0f, -0.0333f, 1.0f, 1.0f, -0.2935f, 0.0f, -0.0264f, 1.0f, 1.0f,
    -0.2722f, 0.0f, -0.0162f, 1.0f, 1.0f, -0.2522f, 0.0f, -0.0029f, 1.0f, 1.0f,
    -0.234f,  0.0f, 0.0132f,  1.0f, 1.0f, -0.2177f, 0.0f, 0.032f,   1.0f, 1.0f,
    -0.2037f, 0.0f, 0.0531f,  1.0f, 1.0f, -0.1922f, 0.0f, 0.0762f,  1.0f, 1.0f,
    -0.1833f, 0.0f, 0.1007f,  1.0f, 1.0f, -0.1774f, 0.0f, 0.1264f,  1.0f, 1.0f,
    -0.1744f, 0.0f, 0.1528f,  1.0f, 1.0f,
};

static const float data11[18 * GP_PRIM_DATABUF_SIZE] = {
    0.963f,  0.0f, 0.1753f, 0.1f,    1.0f, 0.9555f, 0.0f, 0.1761f, 0.2199f, 1.0f,
    0.9367f, 0.0f, 0.1758f, 0.6019f, 1.0f, 0.9202f, 0.0f, 0.1741f, 0.7689f, 1.0f,
    0.9036f, 0.0f, 0.1714f, 0.866f,  1.0f, 0.8885f, 0.0f, 0.1668f, 0.9272f, 1.0f,
    0.8746f, 0.0f, 0.1607f, 0.9653f, 1.0f, 0.8621f, 0.0f, 0.1531f, 0.9876f, 1.0f,
    0.8503f, 0.0f, 0.1447f, 0.9983f, 1.0f, 0.8389f, 0.0f, 0.1352f, 0.999f,  1.0f,
    0.8279f, 0.0f, 0.1244f, 0.9912f, 1.0f, 0.8174f, 0.0f, 0.1125f, 0.9724f, 1.0f,
    0.8079f, 0.0f, 0.099f,  0.9385f, 1.0f, 0.7999f, 0.0f, 0.0839f, 0.8841f, 1.0f,
    0.7935f, 0.0f, 0.0669f, 0.7979f, 1.0f, 0.7892f, 0.0f, 0.0488f, 0.6541f, 1.0f,
    0.787f,  0.0f, 0.0305f, 0.36f,   1.0f, 0.7847f, 0.0f, 0.0139f, 0.1f,    1.0f,
};

static const float data12[18 * GP_PRIM_DATABUF_SIZE] = {
    -1.0227f, 0.0f, 0.1753f, 0.1f,    1.0f, -1.0153f, 0.0f, 0.1761f, 0.2199f, 1.0f,
    -0.9964f, 0.0f, 0.1758f, 0.6019f, 1.0f, -0.9799f, 0.0f, 0.1741f, 0.7689f, 1.0f,
    -0.9634f, 0.0f, 0.1714f, 0.866f,  1.0f, -0.9483f, 0.0f, 0.1668f, 0.9272f, 1.0f,
    -0.9344f, 0.0f, 0.1607f, 0.9653f, 1.0f, -0.9219f, 0.0f, 0.1531f, 0.9876f, 1.0f,
    -0.9101f, 0.0f, 0.1447f, 0.9983f, 1.0f, -0.8986f, 0.0f, 0.1352f, 0.999f,  1.0f,
    -0.8876f, 0.0f, 0.1244f, 0.9912f, 1.0f, -0.8772f, 0.0f, 0.1125f, 0.9724f, 1.0f,
    -0.8677f, 0.0f, 0.099f,  0.9385f, 1.0f, -0.8597f, 0.0f, 0.0839f, 0.8841f, 1.0f,
    -0.8533f, 0.0f, 0.0669f, 0.7979f, 1.0f, -0.849f,  0.0f, 0.0488f, 0.6541f, 1.0f,
    -0.8467f, 0.0f, 0.0305f, 0.36f,   1.0f, -0.8444f, 0.0f, 0.0139f, 0.1f,    1.0f,
};

static const float data13[33 * GP_PRIM_DATABUF_SIZE] = {
    -0.6794f, 0.0f, 0.3908f, 0.1f,    1.0f, -0.6711f, 0.0f, 0.4112f, 0.2199f, 1.0f,
    -0.6513f, 0.0f, 0.4509f, 0.4615f, 1.0f, -0.6276f, 0.0f, 0.489f,  0.6019f, 1.0f,
    -0.6001f, 0.0f, 0.5253f, 0.6981f, 1.0f, -0.5692f, 0.0f, 0.5598f, 0.7689f, 1.0f,
    -0.535f,  0.0f, 0.5924f, 0.8236f, 1.0f, -0.4979f, 0.0f, 0.6228f, 0.866f,  1.0f,
    -0.4579f, 0.0f, 0.651f,  0.9003f, 1.0f, -0.4155f, 0.0f, 0.677f,  0.9272f, 1.0f,
    -0.3707f, 0.0f, 0.7005f, 0.9485f, 1.0f, -0.3239f, 0.0f, 0.7215f, 0.9653f, 1.0f,
    -0.2753f, 0.0f, 0.7399f, 0.9781f, 1.0f, -0.2251f, 0.0f, 0.7555f, 0.9876f, 1.0f,
    -0.1736f, 0.0f, 0.7683f, 0.9942f, 1.0f, -0.121f,  0.0f, 0.778f,  0.9983f, 1.0f,
    -0.0675f, 0.0f, 0.7847f, 0.9997f, 1.0f, -0.0134f, 0.0f, 0.7882f, 0.999f,  1.0f,
    0.0411f,  0.0f, 0.7884f, 0.9963f, 1.0f, 0.0957f,  0.0f, 0.7851f, 0.9912f, 1.0f,
    0.1503f,  0.0f, 0.7783f, 0.9834f, 1.0f, 0.2045f,  0.0f, 0.7678f, 0.9724f, 1.0f,
    0.2581f,  0.0f, 0.7536f, 0.9576f, 1.0f, 0.311f,   0.0f, 0.7355f, 0.9385f, 1.0f,
    0.3628f,  0.0f, 0.7134f, 0.9143f, 1.0f, 0.4133f,  0.0f, 0.6873f, 0.8841f, 1.0f,
    0.4622f,  0.0f, 0.6569f, 0.8461f, 1.0f, 0.5095f,  0.0f, 0.6221f, 0.7979f, 1.0f,
    0.5547f,  0.0f, 0.583f,  0.7359f, 1.0f, 0.5977f,  0.0f, 0.5393f, 0.6541f, 1.0f,
    0.6382f,  0.0f, 0.4909f, 0.5396f, 1.0f, 0.676f,   0.0f, 0.4377f, 0.36f,   1.0f,
    0.7109f,  0.0f, 0.3797f, 0.1f,    1.0f,
};

static const float data14[33 * GP_PRIM_DATABUF_SIZE] = {
    -0.7544f, 0.0f, 0.1799f,  0.1f,    1.0f, -0.7495f, 0.0f, 0.162f,   0.2199f, 1.0f,
    -0.7392f, 0.0f, 0.1283f,  0.4615f, 1.0f, -0.7281f, 0.0f, 0.0975f,  0.6019f, 1.0f,
    -0.7161f, 0.0f, 0.0693f,  0.6981f, 1.0f, -0.7033f, 0.0f, 0.0435f,  0.7689f, 1.0f,
    -0.6898f, 0.0f, 0.02f,    0.8236f, 1.0f, -0.6757f, 0.0f, -0.0014f, 0.866f,  1.0f,
    -0.6609f, 0.0f, -0.0208f, 0.9003f, 1.0f, -0.6455f, 0.0f, -0.0386f, 0.9272f, 1.0f,
    -0.6297f, 0.0f, -0.0547f, 0.9485f, 1.0f, -0.6133f, 0.0f, -0.0695f, 0.9653f, 1.0f,
    -0.5966f, 0.0f, -0.083f,  0.9781f, 1.0f, -0.5795f, 0.0f, -0.0955f, 0.9876f, 1.0f,
    -0.5621f, 0.0f, -0.1071f, 0.9942f, 1.0f, -0.5444f, 0.0f, -0.118f,  0.9983f, 1.0f,
    -0.5265f, 0.0f, -0.1284f, 0.9997f, 1.0f, -0.5084f, 0.0f, -0.1384f, 0.999f,  1.0f,
    -0.4902f, 0.0f, -0.1483f, 0.9963f, 1.0f, -0.4719f, 0.0f, -0.1582f, 0.9912f, 1.0f,
    -0.4537f, 0.0f, -0.1682f, 0.9834f, 1.0f, -0.4355f, 0.0f, -0.1787f, 0.9724f, 1.0f,
    -0.4173f, 0.0f, -0.1896f, 0.9576f, 1.0f, -0.3993f, 0.0f, -0.2013f, 0.9385f, 1.0f,
    -0.3815f, 0.0f, -0.2138f, 0.9143f, 1.0f, -0.364f,  0.0f, -0.2274f, 0.8841f, 1.0f,
    -0.3467f, 0.0f, -0.2422f, 0.8461f, 1.0f, -0.3298f, 0.0f, -0.2584f, 0.7979f, 1.0f,
    -0.3133f, 0.0f, -0.2762f, 0.7359f, 1.0f, -0.2972f, 0.0f, -0.2958f, 0.6541f, 1.0f,
    -0.2816f, 0.0f, -0.3173f, 0.5396f, 1.0f, -0.2665f, 0.0f, -0.3409f, 0.36f,   1.0f,
    -0.2521f, 0.0f, -0.3668f, 0.1f,    1.0f,
};

static const float data15[65 * GP_PRIM_DATABUF_SIZE] = {
    -0.2854f, 0.0f, -0.4528f, 0.1f,    1.0f, -0.2866f, 0.0f, -0.4623f, 0.1288f, 1.0f,
    -0.2899f, 0.0f, -0.4814f, 0.2962f, 1.0f, -0.2943f, 0.0f, -0.5008f, 0.4147f, 1.0f,
    -0.2995f, 0.0f, -0.5203f, 0.5028f, 1.0f, -0.3054f, 0.0f, -0.54f,   0.5723f, 1.0f,
    -0.3117f, 0.0f, -0.5598f, 0.6291f, 1.0f, -0.3182f, 0.0f, -0.5797f, 0.6768f, 1.0f,
    -0.3249f, 0.0f, -0.5996f, 0.7177f, 1.0f, -0.3314f, 0.0f, -0.6196f, 0.753f,  1.0f,
    -0.3376f, 0.0f, -0.6395f, 0.7838f, 1.0f, -0.3432f, 0.0f, -0.6594f, 0.8109f, 1.0f,
    -0.3482f, 0.0f, -0.6792f, 0.8349f, 1.0f, -0.3523f, 0.0f, -0.6989f, 0.8564f, 1.0f,
    -0.3552f, 0.0f, -0.7185f, 0.8756f, 1.0f, -0.3569f, 0.0f, -0.7379f, 0.8922f, 1.0f,
    -0.357f,  0.0f, -0.7571f, 0.9074f, 1.0f, -0.3555f, 0.0f, -0.7761f, 0.9211f, 1.0f,
    -0.3522f, 0.0f, -0.7948f, 0.9329f, 1.0f, -0.3467f, 0.0f, -0.8132f, 0.944f,  1.0f,
    -0.339f,  0.0f, -0.8313f, 0.9531f, 1.0f, -0.3289f, 0.0f, -0.849f,  0.9617f, 1.0f,
    -0.316f,  0.0f, -0.8663f, 0.9688f, 1.0f, -0.3004f, 0.0f, -0.8831f, 0.9755f, 1.0f,
    -0.2817f, 0.0f, -0.8996f, 0.9808f, 1.0f, -0.2598f, 0.0f, -0.9155f, 0.9858f, 1.0f,
    -0.2344f, 0.0f, -0.9309f, 0.9894f, 1.0f, -0.2051f, 0.0f, -0.9457f, 0.993f,  1.0f,
    -0.1716f, 0.0f, -0.9599f, 0.9952f, 1.0f, -0.1341f, 0.0f, -0.9733f, 0.9973f, 1.0f,
    -0.0928f, 0.0f, -0.9857f, 0.9987f, 1.0f, -0.05f,   0.0f, -0.9962f, 0.9993f, 1.0f,
    -0.0087f, 0.0f, -1.0041f, 1.0f,    1.0f, 0.0287f,  0.0f, -1.0087f, 0.9993f, 1.0f,
    0.062f,   0.0f, -1.0104f, 0.9987f, 1.0f, 0.0924f,  0.0f, -1.0102f, 0.9973f, 1.0f,
    0.1205f,  0.0f, -1.0086f, 0.9952f, 1.0f, 0.1465f,  0.0f, -1.0057f, 0.993f,  1.0f,
    0.1706f,  0.0f, -1.0017f, 0.9894f, 1.0f, 0.1928f,  0.0f, -0.9964f, 0.9858f, 1.0f,
    0.2132f,  0.0f, -0.99f,   0.9808f, 1.0f, 0.2318f,  0.0f, -0.9824f, 0.9755f, 1.0f,
    0.2487f,  0.0f, -0.9738f, 0.9688f, 1.0f, 0.2641f,  0.0f, -0.9641f, 0.9617f, 1.0f,
    0.2778f,  0.0f, -0.9533f, 0.9531f, 1.0f, 0.2901f,  0.0f, -0.9415f, 0.944f,  1.0f,
    0.3009f,  0.0f, -0.9287f, 0.9329f, 1.0f, 0.3103f,  0.0f, -0.9148f, 0.9211f, 1.0f,
    0.3183f,  0.0f, -0.8999f, 0.9074f, 1.0f, 0.325f,   0.0f, -0.8841f, 0.8922f, 1.0f,
    0.3304f,  0.0f, -0.8672f, 0.8756f, 1.0f, 0.3345f,  0.0f, -0.8493f, 0.8564f, 1.0f,
    0.3374f,  0.0f, -0.8305f, 0.8349f, 1.0f, 0.3391f,  0.0f, -0.8107f, 0.8109f, 1.0f,
    0.3397f,  0.0f, -0.7899f, 0.7838f, 1.0f, 0.3392f,  0.0f, -0.7682f, 0.753f,  1.0f,
    0.3377f,  0.0f, -0.7456f, 0.7177f, 1.0f, 0.3352f,  0.0f, -0.7221f, 0.6768f, 1.0f,
    0.3317f,  0.0f, -0.6976f, 0.6291f, 1.0f, 0.3273f,  0.0f, -0.6722f, 0.5723f, 1.0f,
    0.322f,   0.0f, -0.646f,  0.5028f, 1.0f, 0.316f,   0.0f, -0.6188f, 0.4147f, 1.0f,
    0.3091f,  0.0f, -0.5908f, 0.2962f, 1.0f, 0.3015f,  0.0f, -0.5619f, 0.1288f, 1.0f,
    0.2974f,  0.0f, -0.5472f, 0.1f,    1.0f,
};

static const float data16[34 * GP_PRIM_DATABUF_SIZE] = {
    -0.4408f, 0.0f, 0.5073f, 0.1f,    1.0f, -0.4312f, 0.0f, 0.5132f, 0.1288f, 1.0f,
    -0.3945f, 0.0f, 0.5319f, 0.4147f, 1.0f, -0.3605f, 0.0f, 0.5438f, 0.5723f, 1.0f,
    -0.3288f, 0.0f, 0.5496f, 0.6768f, 1.0f, -0.2992f, 0.0f, 0.5501f, 0.753f,  1.0f,
    -0.2712f, 0.0f, 0.546f,  0.8109f, 1.0f, -0.2446f, 0.0f, 0.5381f, 0.8564f, 1.0f,
    -0.2191f, 0.0f, 0.5272f, 0.8922f, 1.0f, -0.1943f, 0.0f, 0.5139f, 0.9211f, 1.0f,
    -0.1699f, 0.0f, 0.4992f, 0.944f,  1.0f, -0.1456f, 0.0f, 0.4837f, 0.9617f, 1.0f,
    -0.1211f, 0.0f, 0.4682f, 0.9755f, 1.0f, -0.096f,  0.0f, 0.4535f, 0.9858f, 1.0f,
    -0.07f,   0.0f, 0.4404f, 0.993f,  1.0f, -0.0428f, 0.0f, 0.4301f, 0.9973f, 1.0f,
    -0.016f,  0.0f, 0.4235f, 0.9993f, 1.0f, 0.0076f,  0.0f, 0.4216f, 0.9993f, 1.0f,
    0.0283f,  0.0f, 0.4237f, 0.9973f, 1.0f, 0.0477f,  0.0f, 0.4291f, 0.993f,  1.0f,
    0.067f,   0.0f, 0.4369f, 0.9858f, 1.0f, 0.0866f,  0.0f, 0.4465f, 0.9755f, 1.0f,
    0.1068f,  0.0f, 0.4576f, 0.9617f, 1.0f, 0.1278f,  0.0f, 0.4695f, 0.944f,  1.0f,
    0.1498f,  0.0f, 0.4819f, 0.9211f, 1.0f, 0.173f,   0.0f, 0.4941f, 0.8922f, 1.0f,
    0.1977f,  0.0f, 0.5056f, 0.8564f, 1.0f, 0.2241f,  0.0f, 0.5159f, 0.8109f, 1.0f,
    0.2523f,  0.0f, 0.5244f, 0.753f,  1.0f, 0.2825f,  0.0f, 0.5307f, 0.6768f, 1.0f,
    0.315f,   0.0f, 0.5342f, 0.5723f, 1.0f, 0.3499f,  0.0f, 0.5345f, 0.4147f, 1.0f,
    0.3875f,  0.0f, 0.5311f, 0.1288f, 1.0f, 0.3973f,  0.0f, 0.5295f, 0.1f,    1.0f,
};

static const float data17[33 * GP_PRIM_DATABUF_SIZE] = {
    0.724f,  0.0f, 0.1799f,  0.1f,    1.0f, 0.7192f, 0.0f, 0.162f,   0.2199f, 1.0f,
    0.7089f, 0.0f, 0.1283f,  0.4615f, 1.0f, 0.6977f, 0.0f, 0.0975f,  0.6019f, 1.0f,
    0.6858f, 0.0f, 0.0693f,  0.6981f, 1.0f, 0.673f,  0.0f, 0.0435f,  0.7689f, 1.0f,
    0.6595f, 0.0f, 0.02f,    0.8236f, 1.0f, 0.6453f, 0.0f, -0.0014f, 0.866f,  1.0f,
    0.6306f, 0.0f, -0.0208f, 0.9003f, 1.0f, 0.6152f, 0.0f, -0.0386f, 0.9272f, 1.0f,
    0.5994f, 0.0f, -0.0547f, 0.9485f, 1.0f, 0.583f,  0.0f, -0.0695f, 0.9653f, 1.0f,
    0.5663f, 0.0f, -0.083f,  0.9781f, 1.0f, 0.5492f, 0.0f, -0.0955f, 0.9876f, 1.0f,
    0.5317f, 0.0f, -0.1071f, 0.9942f, 1.0f, 0.514f,  0.0f, -0.118f,  0.9983f, 1.0f,
    0.4961f, 0.0f, -0.1284f, 0.9997f, 1.0f, 0.4781f, 0.0f, -0.1384f, 0.999f,  1.0f,
    0.4599f, 0.0f, -0.1483f, 0.9963f, 1.0f, 0.4416f, 0.0f, -0.1582f, 0.9912f, 1.0f,
    0.4234f, 0.0f, -0.1682f, 0.9834f, 1.0f, 0.4051f, 0.0f, -0.1787f, 0.9724f, 1.0f,
    0.387f,  0.0f, -0.1896f, 0.9576f, 1.0f, 0.369f,  0.0f, -0.2013f, 0.9385f, 1.0f,
    0.3512f, 0.0f, -0.2138f, 0.9143f, 1.0f, 0.3337f, 0.0f, -0.2274f, 0.8841f, 1.0f,
    0.3164f, 0.0f, -0.2422f, 0.8461f, 1.0f, 0.2995f, 0.0f, -0.2584f, 0.7979f, 1.0f,
    0.2829f, 0.0f, -0.2762f, 0.7359f, 1.0f, 0.2669f, 0.0f, -0.2958f, 0.6541f, 1.0f,
    0.2513f, 0.0f, -0.3173f, 0.5396f, 1.0f, 0.2362f, 0.0f, -0.3409f, 0.36f,   1.0f,
    0.2218f, 0.0f, -0.3668f, 0.1f,    1.0f,
};

static const float data18[33 * GP_PRIM_DATABUF_SIZE] = {
    -0.1119f, 0.0f, -0.7868f, 0.1f,    1.0f, -0.1087f, 0.0f, -0.7905f, 0.2199f, 1.0f,
    -0.1022f, 0.0f, -0.7974f, 0.4615f, 1.0f, -0.0955f, 0.0f, -0.8038f, 0.6019f, 1.0f,
    -0.0886f, 0.0f, -0.8097f, 0.6981f, 1.0f, -0.0816f, 0.0f, -0.8152f, 0.7689f, 1.0f,
    -0.0744f, 0.0f, -0.8201f, 0.8236f, 1.0f, -0.0671f, 0.0f, -0.8246f, 0.866f,  1.0f,
    -0.0597f, 0.0f, -0.8286f, 0.9003f, 1.0f, -0.0522f, 0.0f, -0.8321f, 0.9272f, 1.0f,
    -0.0446f, 0.0f, -0.8351f, 0.9485f, 1.0f, -0.0369f, 0.0f, -0.8376f, 0.9653f, 1.0f,
    -0.0292f, 0.0f, -0.8397f, 0.9781f, 1.0f, -0.0214f, 0.0f, -0.8413f, 0.9876f, 1.0f,
    -0.0136f, 0.0f, -0.8424f, 0.9942f, 1.0f, -0.0057f, 0.0f, -0.843f,  0.9983f, 1.0f,
    0.0022f,  0.0f, -0.8431f, 0.9997f, 1.0f, 0.01f,    0.0f, -0.8428f, 0.999f,  1.0f,
    0.0178f,  0.0f, -0.8419f, 0.9963f, 1.0f, 0.0257f,  0.0f, -0.8407f, 0.9912f, 1.0f,
    0.0334f,  0.0f, -0.8389f, 0.9834f, 1.0f, 0.0411f,  0.0f, -0.8366f, 0.9724f, 1.0f,
    0.0488f,  0.0f, -0.8339f, 0.9576f, 1.0f, 0.0564f,  0.0f, -0.8308f, 0.9385f, 1.0f,
    0.0638f,  0.0f, -0.8271f, 0.9143f, 1.0f, 0.0712f,  0.0f, -0.823f,  0.8841f, 1.0f,
    0.0785f,  0.0f, -0.8184f, 0.8461f, 1.0f, 0.0856f,  0.0f, -0.8134f, 0.7979f, 1.0f,
    0.0925f,  0.0f, -0.8079f, 0.7359f, 1.0f, 0.0994f,  0.0f, -0.8019f, 0.6541f, 1.0f,
    0.106f,   0.0f, -0.7954f, 0.5396f, 1.0f, 0.1125f,  0.0f, -0.7885f, 0.36f,   1.0f,
    0.1187f,  0.0f, -0.7812f, 0.1f,    1.0f,
};

static const float data19[34 * GP_PRIM_DATABUF_SIZE] = {
    -0.1119f, 0.0f, -0.2655f, 0.1f,    1.0f, -0.1101f, 0.0f, -0.2605f, 0.1288f, 1.0f,
    -0.1032f, 0.0f, -0.2424f, 0.4147f, 1.0f, -0.0965f, 0.0f, -0.2276f, 0.5723f, 1.0f,
    -0.0901f, 0.0f, -0.2158f, 0.6768f, 1.0f, -0.0837f, 0.0f, -0.2069f, 0.753f,  1.0f,
    -0.0775f, 0.0f, -0.2006f, 0.8109f, 1.0f, -0.0712f, 0.0f, -0.1967f, 0.8564f, 1.0f,
    -0.065f,  0.0f, -0.1949f, 0.8922f, 1.0f, -0.0587f, 0.0f, -0.195f,  0.9211f, 1.0f,
    -0.0522f, 0.0f, -0.1968f, 0.944f,  1.0f, -0.0455f, 0.0f, -0.2001f, 0.9617f, 1.0f,
    -0.0386f, 0.0f, -0.2046f, 0.9755f, 1.0f, -0.0315f, 0.0f, -0.21f,   0.9858f, 1.0f,
    -0.0239f, 0.0f, -0.2163f, 0.993f,  1.0f, -0.016f,  0.0f, -0.223f,  0.9973f, 1.0f,
    -0.0075f, 0.0f, -0.2301f, 0.9993f, 1.0f, -0.0002f, 0.0f, -0.2342f, 0.9993f, 1.0f,
    0.0054f,  0.0f, -0.234f,  0.9973f, 1.0f, 0.0106f,  0.0f, -0.2324f, 0.993f,  1.0f,
    0.0155f,  0.0f, -0.2297f, 0.9858f, 1.0f, 0.0202f,  0.0f, -0.2261f, 0.9755f, 1.0f,
    0.0249f,  0.0f, -0.2219f, 0.9617f, 1.0f, 0.0296f,  0.0f, -0.2174f, 0.944f,  1.0f,
    0.0344f,  0.0f, -0.2128f, 0.9211f, 1.0f, 0.0394f,  0.0f, -0.2084f, 0.8922f, 1.0f,
    0.0448f,  0.0f, -0.2044f, 0.8564f, 1.0f, 0.0506f,  0.0f, -0.2012f, 0.8109f, 1.0f,
    0.057f,   0.0f, -0.1988f, 0.753f,  1.0f, 0.064f,   0.0f, -0.1977f, 0.6768f, 1.0f,
    0.0717f,  0.0f, -0.198f,  0.5723f, 1.0f, 0.0804f,  0.0f, -0.2001f, 0.4147f, 1.0f,
    0.0899f,  0.0f, -0.2042f, 0.1288f, 1.0f, 0.0925f,  0.0f, -0.2055f, 0.1f,    1.0f,
};

static const float data20[33 * GP_PRIM_DATABUF_SIZE] = {
    0.7993f, 0.0f, 0.3242f,  0.1f,    1.0f, 0.8169f, 0.0f, 0.3354f,  0.2199f, 1.0f,
    0.8512f, 0.0f, 0.3542f,  0.4615f, 1.0f, 0.884f,  0.0f, 0.3686f,  0.6019f, 1.0f,
    0.9153f, 0.0f, 0.3787f,  0.6981f, 1.0f, 0.945f,  0.0f, 0.3848f,  0.7689f, 1.0f,
    0.9729f, 0.0f, 0.3871f,  0.8236f, 1.0f, 0.9989f, 0.0f, 0.3857f,  0.866f,  1.0f,
    1.0229f, 0.0f, 0.3811f,  0.9003f, 1.0f, 1.0447f, 0.0f, 0.3732f,  0.9272f, 1.0f,
    1.0643f, 0.0f, 0.3624f,  0.9485f, 1.0f, 1.0814f, 0.0f, 0.349f,   0.9653f, 1.0f,
    1.096f,  0.0f, 0.333f,   0.9781f, 1.0f, 1.108f,  0.0f, 0.3147f,  0.9876f, 1.0f,
    1.1172f, 0.0f, 0.2944f,  0.9942f, 1.0f, 1.1235f, 0.0f, 0.2723f,  0.9983f, 1.0f,
    1.1267f, 0.0f, 0.2485f,  0.9997f, 1.0f, 1.1269f, 0.0f, 0.2233f,  0.999f,  1.0f,
    1.1237f, 0.0f, 0.197f,   0.9963f, 1.0f, 1.1172f, 0.0f, 0.1697f,  0.9912f, 1.0f,
    1.1071f, 0.0f, 0.1417f,  0.9834f, 1.0f, 1.0934f, 0.0f, 0.1131f,  0.9724f, 1.0f,
    1.0759f, 0.0f, 0.0842f,  0.9576f, 1.0f, 1.0545f, 0.0f, 0.0553f,  0.9385f, 1.0f,
    1.0291f, 0.0f, 0.0265f,  0.9143f, 1.0f, 0.9995f, 0.0f, -0.0019f, 0.8841f, 1.0f,
    0.9657f, 0.0f, -0.0298f, 0.8461f, 1.0f, 0.9275f, 0.0f, -0.0569f, 0.7979f, 1.0f,
    0.8847f, 0.0f, -0.0829f, 0.7359f, 1.0f, 0.8373f, 0.0f, -0.1077f, 0.6541f, 1.0f,
    0.7852f, 0.0f, -0.1311f, 0.5396f, 1.0f, 0.7281f, 0.0f, -0.1528f, 0.36f,   1.0f,
    0.6661f, 0.0f, -0.1725f, 0.1f,    1.0f,
};

static const float data21[64 * GP_PRIM_DATABUF_SIZE] = {
    -0.7428f, 0.0f, 0.2977f,  0.1f,    1.0f, -0.7608f, 0.0f, 0.3138f,  0.2199f, 1.0f,
    -0.7786f, 0.0f, 0.3284f,  0.36f,   1.0f, -0.7962f, 0.0f, 0.3414f,  0.4615f, 1.0f,
    -0.8135f, 0.0f, 0.3531f,  0.5396f, 1.0f, -0.8306f, 0.0f, 0.3633f,  0.6019f, 1.0f,
    -0.8473f, 0.0f, 0.3722f,  0.6541f, 1.0f, -0.8637f, 0.0f, 0.3798f,  0.6981f, 1.0f,
    -0.8798f, 0.0f, 0.386f,   0.7359f, 1.0f, -0.8956f, 0.0f, 0.3911f,  0.7689f, 1.0f,
    -0.9109f, 0.0f, 0.3949f,  0.7979f, 1.0f, -0.9259f, 0.0f, 0.3975f,  0.8236f, 1.0f,
    -0.9405f, 0.0f, 0.3989f,  0.8461f, 1.0f, -0.9546f, 0.0f, 0.3993f,  0.866f,  1.0f,
    -0.9682f, 0.0f, 0.3986f,  0.8841f, 1.0f, -0.9814f, 0.0f, 0.3969f,  0.9003f, 1.0f,
    -0.994f,  0.0f, 0.3941f,  0.9143f, 1.0f, -1.0062f, 0.0f, 0.3904f,  0.9272f, 1.0f,
    -1.0178f, 0.0f, 0.3858f,  0.9385f, 1.0f, -1.0288f, 0.0f, 0.3803f,  0.9485f, 1.0f,
    -1.0393f, 0.0f, 0.3739f,  0.9576f, 1.0f, -1.0492f, 0.0f, 0.3668f,  0.9653f, 1.0f,
    -1.0584f, 0.0f, 0.3588f,  0.9724f, 1.0f, -1.067f,  0.0f, 0.3501f,  0.9781f, 1.0f,
    -1.075f,  0.0f, 0.3407f,  0.9834f, 1.0f, -1.0822f, 0.0f, 0.3307f,  0.9876f, 1.0f,
    -1.0888f, 0.0f, 0.32f,    0.9912f, 1.0f, -1.0946f, 0.0f, 0.3087f,  0.9942f, 1.0f,
    -1.0997f, 0.0f, 0.2969f,  0.9963f, 1.0f, -1.104f,  0.0f, 0.2845f,  0.9983f, 1.0f,
    -1.1075f, 0.0f, 0.2717f,  0.999f,  1.0f, -1.1103f, 0.0f, 0.2584f,  0.9997f, 1.0f,
    -1.1122f, 0.0f, 0.2447f,  0.9997f, 1.0f, -1.1132f, 0.0f, 0.2306f,  0.999f,  1.0f,
    -1.1134f, 0.0f, 0.2162f,  0.9983f, 1.0f, -1.1128f, 0.0f, 0.2015f,  0.9963f, 1.0f,
    -1.1112f, 0.0f, 0.1865f,  0.9942f, 1.0f, -1.1086f, 0.0f, 0.1713f,  0.9912f, 1.0f,
    -1.1052f, 0.0f, 0.1559f,  0.9876f, 1.0f, -1.1007f, 0.0f, 0.1404f,  0.9834f, 1.0f,
    -1.0953f, 0.0f, 0.1247f,  0.9781f, 1.0f, -1.0889f, 0.0f, 0.109f,   0.9724f, 1.0f,
    -1.0814f, 0.0f, 0.0932f,  0.9653f, 1.0f, -1.0729f, 0.0f, 0.0774f,  0.9576f, 1.0f,
    -1.0633f, 0.0f, 0.0617f,  0.9485f, 1.0f, -1.0527f, 0.0f, 0.046f,   0.9385f, 1.0f,
    -1.0409f, 0.0f, 0.0304f,  0.9272f, 1.0f, -1.028f,  0.0f, 0.015f,   0.9143f, 1.0f,
    -1.0139f, 0.0f, -0.0003f, 0.9003f, 1.0f, -0.9987f, 0.0f, -0.0154f, 0.8841f, 1.0f,
    -0.9823f, 0.0f, -0.0302f, 0.866f,  1.0f, -0.9646f, 0.0f, -0.0447f, 0.8461f, 1.0f,
    -0.9458f, 0.0f, -0.0589f, 0.8236f, 1.0f, -0.9257f, 0.0f, -0.0727f, 0.7979f, 1.0f,
    -0.9043f, 0.0f, -0.0862f, 0.7689f, 1.0f, -0.8816f, 0.0f, -0.0992f, 0.7359f, 1.0f,
    -0.8576f, 0.0f, -0.1117f, 0.6981f, 1.0f, -0.8323f, 0.0f, -0.1237f, 0.6541f, 1.0f,
    -0.8056f, 0.0f, -0.1352f, 0.6019f, 1.0f, -0.7775f, 0.0f, -0.1461f, 0.5396f, 1.0f,
    -0.7481f, 0.0f, -0.1564f, 0.4615f, 1.0f, -0.7172f, 0.0f, -0.166f,  0.36f,   1.0f,
    -0.6849f, 0.0f, -0.175f,  0.2199f, 1.0f, -0.6512f, 0.0f, -0.1832f, 0.1f,    1.0f,
};

static const float data22[26 * GP_PRIM_DATABUF_SIZE] = {
    0.2721f, 0.0f, 0.2084f, 1.0f, 1.0f, 0.2721f, 0.0f, 0.2112f, 1.0f, 1.0f,
    0.2739f, 0.0f, 0.2223f, 1.0f, 1.0f, 0.2778f, 0.0f, 0.2324f, 1.0f, 1.0f,
    0.2836f, 0.0f, 0.2409f, 1.0f, 1.0f, 0.291f,  0.0f, 0.2471f, 1.0f, 1.0f,
    0.2994f, 0.0f, 0.2507f, 1.0f, 1.0f, 0.3082f, 0.0f, 0.2515f, 1.0f, 1.0f,
    0.3169f, 0.0f, 0.2493f, 1.0f, 1.0f, 0.3248f, 0.0f, 0.2443f, 1.0f, 1.0f,
    0.3315f, 0.0f, 0.2369f, 1.0f, 1.0f, 0.3364f, 0.0f, 0.2275f, 1.0f, 1.0f,
    0.3392f, 0.0f, 0.2168f, 1.0f, 1.0f, 0.3398f, 0.0f, 0.2056f, 1.0f, 1.0f,
    0.3381f, 0.0f, 0.1945f, 1.0f, 1.0f, 0.3342f, 0.0f, 0.1844f, 1.0f, 1.0f,
    0.3283f, 0.0f, 0.176f,  1.0f, 1.0f, 0.321f,  0.0f, 0.1697f, 1.0f, 1.0f,
    0.3126f, 0.0f, 0.1661f, 1.0f, 1.0f, 0.3038f, 0.0f, 0.1653f, 1.0f, 1.0f,
    0.2951f, 0.0f, 0.1675f, 1.0f, 1.0f, 0.2871f, 0.0f, 0.1725f, 1.0f, 1.0f,
    0.2805f, 0.0f, 0.1799f, 1.0f, 1.0f, 0.2756f, 0.0f, 0.1893f, 1.0f, 1.0f,
    0.2727f, 0.0f, 0.2f,    1.0f, 1.0f, 0.2721f, 0.0f, 0.2056f, 1.0f, 1.0f,
};

static const float data23[26 * GP_PRIM_DATABUF_SIZE] = {
    -0.3545f, 0.0f, 0.2084f, 1.0f, 1.0f, -0.3544f, 0.0f, 0.2112f, 1.0f, 1.0f,
    -0.3527f, 0.0f, 0.2223f, 1.0f, 1.0f, -0.3488f, 0.0f, 0.2324f, 1.0f, 1.0f,
    -0.343f,  0.0f, 0.2409f, 1.0f, 1.0f, -0.3356f, 0.0f, 0.2471f, 1.0f, 1.0f,
    -0.3272f, 0.0f, 0.2507f, 1.0f, 1.0f, -0.3184f, 0.0f, 0.2515f, 1.0f, 1.0f,
    -0.3097f, 0.0f, 0.2493f, 1.0f, 1.0f, -0.3018f, 0.0f, 0.2443f, 1.0f, 1.0f,
    -0.2951f, 0.0f, 0.2369f, 1.0f, 1.0f, -0.2902f, 0.0f, 0.2275f, 1.0f, 1.0f,
    -0.2874f, 0.0f, 0.2168f, 1.0f, 1.0f, -0.2868f, 0.0f, 0.2056f, 1.0f, 1.0f,
    -0.2885f, 0.0f, 0.1945f, 1.0f, 1.0f, -0.2924f, 0.0f, 0.1844f, 1.0f, 1.0f,
    -0.2983f, 0.0f, 0.176f,  1.0f, 1.0f, -0.3056f, 0.0f, 0.1697f, 1.0f, 1.0f,
    -0.314f,  0.0f, 0.1661f, 1.0f, 1.0f, -0.3228f, 0.0f, 0.1653f, 1.0f, 1.0f,
    -0.3315f, 0.0f, 0.1675f, 1.0f, 1.0f, -0.3394f, 0.0f, 0.1725f, 1.0f, 1.0f,
    -0.3461f, 0.0f, 0.1799f, 1.0f, 1.0f, -0.351f,  0.0f, 0.1893f, 1.0f, 1.0f,
    -0.3539f, 0.0f, 0.2f,    1.0f, 1.0f, -0.3544f, 0.0f, 0.2056f, 1.0f, 1.0f,
};

static const float data24[33 * GP_PRIM_DATABUF_SIZE] = {
    0.1904f, 0.0f, 0.0983f,  0.1f,    1.0f, 0.1997f, 0.0f, 0.0774f,  0.2199f, 1.0f,
    0.2234f, 0.0f, 0.0407f,  0.4615f, 1.0f, 0.2477f, 0.0f, 0.0158f,  0.6019f, 1.0f,
    0.2767f, 0.0f, -0.0047f, 0.6981f, 1.0f, 0.3061f, 0.0f, -0.0179f, 0.7689f, 1.0f,
    0.3346f, 0.0f, -0.0247f, 0.8236f, 1.0f, 0.362f,  0.0f, -0.0262f, 0.866f,  1.0f,
    0.3881f, 0.0f, -0.0231f, 0.9003f, 1.0f, 0.4128f, 0.0f, -0.0159f, 0.9272f, 1.0f,
    0.436f,  0.0f, -0.0049f, 0.9485f, 1.0f, 0.4577f, 0.0f, 0.0099f,  0.9653f, 1.0f,
    0.4778f, 0.0f, 0.0289f,  0.9781f, 1.0f, 0.4959f, 0.0f, 0.0524f,  0.9876f, 1.0f,
    0.5114f, 0.0f, 0.0806f,  0.9942f, 1.0f, 0.5229f, 0.0f, 0.1134f,  0.9983f, 1.0f,
    0.5291f, 0.0f, 0.1498f,  0.9997f, 1.0f, 0.5289f, 0.0f, 0.1884f,  0.999f,  1.0f,
    0.5219f, 0.0f, 0.2267f,  0.9963f, 1.0f, 0.5087f, 0.0f, 0.2616f,  0.9912f, 1.0f,
    0.4907f, 0.0f, 0.2916f,  0.9834f, 1.0f, 0.4697f, 0.0f, 0.3159f,  0.9724f, 1.0f,
    0.4465f, 0.0f, 0.3349f,  0.9576f, 1.0f, 0.422f,  0.0f, 0.3489f,  0.9385f, 1.0f,
    0.3964f, 0.0f, 0.3582f,  0.9143f, 1.0f, 0.3705f, 0.0f, 0.3628f,  0.8841f, 1.0f,
    0.3436f, 0.0f, 0.363f,   0.8461f, 1.0f, 0.3166f, 0.0f, 0.3586f,  0.7979f, 1.0f,
    0.2897f, 0.0f, 0.3491f,  0.7359f, 1.0f, 0.2635f, 0.0f, 0.3344f,  0.6541f, 1.0f,
    0.2382f, 0.0f, 0.3137f,  0.5396f, 1.0f, 0.2162f, 0.0f, 0.2882f,  0.36f,   1.0f,
    0.1949f, 0.0f, 0.2516f,  0.1f,    1.0f,
};

static const float data25[18 * GP_PRIM_DATABUF_SIZE] = {
    -1.0361f, 0.0f, 0.1745f, 0.1f,    1.0f, -1.0319f, 0.0f, 0.177f,  0.2199f, 1.0f,
    -1.0153f, 0.0f, 0.1852f, 0.6019f, 1.0f, -0.999f,  0.0f, 0.1907f, 0.7689f, 1.0f,
    -0.983f,  0.0f, 0.1936f, 0.866f,  1.0f, -0.9673f, 0.0f, 0.1941f, 0.9272f, 1.0f,
    -0.9522f, 0.0f, 0.1923f, 0.9653f, 1.0f, -0.9376f, 0.0f, 0.1885f, 0.9876f, 1.0f,
    -0.9236f, 0.0f, 0.1827f, 0.9983f, 1.0f, -0.9103f, 0.0f, 0.1752f, 0.999f,  1.0f,
    -0.8978f, 0.0f, 0.1662f, 0.9912f, 1.0f, -0.8862f, 0.0f, 0.1558f, 0.9724f, 1.0f,
    -0.8756f, 0.0f, 0.1441f, 0.9385f, 1.0f, -0.866f,  0.0f, 0.1314f, 0.8841f, 1.0f,
    -0.8575f, 0.0f, 0.1179f, 0.7979f, 1.0f, -0.8501f, 0.0f, 0.1036f, 0.6541f, 1.0f,
    -0.8441f, 0.0f, 0.0889f, 0.36f,   1.0f, -0.8416f, 0.0f, 0.0813f, 0.1f,    1.0f,
};

static const float data26[18 * GP_PRIM_DATABUF_SIZE] = {
    0.9747f, 0.0f, 0.1745f, 0.1f,    1.0f, 0.9706f, 0.0f, 0.177f,  0.2199f, 1.0f,
    0.954f,  0.0f, 0.1852f, 0.6019f, 1.0f, 0.9377f, 0.0f, 0.1907f, 0.7689f, 1.0f,
    0.9216f, 0.0f, 0.1936f, 0.866f,  1.0f, 0.906f,  0.0f, 0.1941f, 0.9272f, 1.0f,
    0.8908f, 0.0f, 0.1923f, 0.9653f, 1.0f, 0.8762f, 0.0f, 0.1885f, 0.9876f, 1.0f,
    0.8622f, 0.0f, 0.1827f, 0.9983f, 1.0f, 0.849f,  0.0f, 0.1752f, 0.999f,  1.0f,
    0.8365f, 0.0f, 0.1662f, 0.9912f, 1.0f, 0.8249f, 0.0f, 0.1558f, 0.9724f, 1.0f,
    0.8143f, 0.0f, 0.1441f, 0.9385f, 1.0f, 0.8046f, 0.0f, 0.1314f, 0.8841f, 1.0f,
    0.7961f, 0.0f, 0.1179f, 0.7979f, 1.0f, 0.7888f, 0.0f, 0.1036f, 0.6541f, 1.0f,
    0.7828f, 0.0f, 0.0889f, 0.36f,   1.0f, 0.7802f, 0.0f, 0.0813f, 0.1f,    1.0f,
};

static const float data27[33 * GP_PRIM_DATABUF_SIZE] = {
    -0.186f,  0.0f, 0.0983f,  0.1f,    1.0f, -0.1954f, 0.0f, 0.0774f,  0.2199f, 1.0f,
    -0.2191f, 0.0f, 0.0407f,  0.4615f, 1.0f, -0.2434f, 0.0f, 0.0158f,  0.6019f, 1.0f,
    -0.2723f, 0.0f, -0.0047f, 0.6981f, 1.0f, -0.3017f, 0.0f, -0.0179f, 0.7689f, 1.0f,
    -0.3302f, 0.0f, -0.0247f, 0.8236f, 1.0f, -0.3576f, 0.0f, -0.0262f, 0.866f,  1.0f,
    -0.3837f, 0.0f, -0.0231f, 0.9003f, 1.0f, -0.4085f, 0.0f, -0.0159f, 0.9272f, 1.0f,
    -0.4317f, 0.0f, -0.0049f, 0.9485f, 1.0f, -0.4533f, 0.0f, 0.0099f,  0.9653f, 1.0f,
    -0.4734f, 0.0f, 0.0289f,  0.9781f, 1.0f, -0.4916f, 0.0f, 0.0524f,  0.9876f, 1.0f,
    -0.507f,  0.0f, 0.0806f,  0.9942f, 1.0f, -0.5185f, 0.0f, 0.1134f,  0.9983f, 1.0f,
    -0.5247f, 0.0f, 0.1498f,  0.9997f, 1.0f, -0.5246f, 0.0f, 0.1884f,  0.999f,  1.0f,
    -0.5175f, 0.0f, 0.2267f,  0.9963f, 1.0f, -0.5043f, 0.0f, 0.2616f,  0.9912f, 1.0f,
    -0.4864f, 0.0f, 0.2916f,  0.9834f, 1.0f, -0.4653f, 0.0f, 0.3159f,  0.9724f, 1.0f,
    -0.4422f, 0.0f, 0.3349f,  0.9576f, 1.0f, -0.4177f, 0.0f, 0.3489f,  0.9385f, 1.0f,
    -0.3921f, 0.0f, 0.3582f,  0.9143f, 1.0f, -0.3661f, 0.0f, 0.3628f,  0.8841f, 1.0f,
    -0.3392f, 0.0f, 0.363f,   0.8461f, 1.0f, -0.3123f, 0.0f, 0.3586f,  0.7979f, 1.0f,
    -0.2853f, 0.0f, 0.3491f,  0.7359f, 1.0f, -0.2591f, 0.0f, 0.3344f,  0.6541f, 1.0f,
    -0.2339f, 0.0f, 0.3137f,  0.5396f, 1.0f, -0.2118f, 0.0f, 0.2882f,  0.36f,   1.0f,
    -0.1906f, 0.0f, 0.2516f,  0.1f,    1.0f,
};

/* ***************************************************************** */
/* Monkey Color Data */

static const ColorTemplate gp_monkey_pct_black = {
    N_("Black"),
    {0.0f, 0.0f, 0.0f, 1.0f},
    {0.0f, 0.0f, 0.0f, 0.0f},
};

static const ColorTemplate gp_monkey_pct_skin = {
    N_("Skin"),
    {0.733f, 0.569f, 0.361f, 1.0f},
    {0.745f, 0.502f, 0.278f, 1.0f},
};

static const ColorTemplate gp_monkey_pct_skin_light = {
    N_("Skin_Light"),
    {0.914f, 0.827f, 0.635f, 1.0f},
    {0.913f, 0.828f, 0.637f, 0.0f},
};

static const ColorTemplate gp_monkey_pct_skin_shadow = {
    N_("Skin_Shadow"),
    {0.322f, 0.29f, 0.224f, 0.5f},
    {0.32f, 0.29f, 0.223f, 0.3f},
};

static const ColorTemplate gp_monkey_pct_eyes = {
    N_("Eyes"),
    {0.553f, 0.39f, 0.266f, 0.0f},
    {0.847f, 0.723f, 0.599f, 1.0f},
};

static const ColorTemplate gp_monkey_pct_pupils = {
    N_("Pupils"),
    {0.0f, 0.0f, 0.0f, 0.0f},
    {0.0f, 0.0f, 0.0f, 1.0f},
};

/* ***************************************************************** */
/* Monkey API */

void ED_gpencil_create_monkey(bContext *C, Object *ob, float mat[4][4])
{
  /* Original model created by Matias Mendiola. */

  Main *bmain = CTX_data_main(C);
  Scene *scene = CTX_data_scene(C);
  bGPdata *gpd = (bGPdata *)ob->data;
  bGPDstroke *gps;

  /* create colors */
  int color_Black = gpencil_monkey_color(bmain, ob, &gp_monkey_pct_black, true, false);
  int color_Skin = gpencil_monkey_color(bmain, ob, &gp_monkey_pct_skin, false, true);
  int color_Skin_Light = gpencil_monkey_color(bmain, ob, &gp_monkey_pct_skin_light, true, false);
  int color_Skin_Shadow = gpencil_monkey_color(bmain, ob, &gp_monkey_pct_skin_shadow, true, false);
  int color_Eyes = gpencil_monkey_color(bmain, ob, &gp_monkey_pct_eyes, false, true);
  int color_Pupils = gpencil_monkey_color(bmain, ob, &gp_monkey_pct_pupils, false, true);

  /* set first color as active */
  ob->actcol = color_Black + 1;

  /* layers */
  /* NOTE: For now, we just add new layers, to make it easier to separate out old/new instances */
  bGPDlayer *Fills = BKE_gpencil_layer_addnew(gpd, "Fills", false, false);
  bGPDlayer *Lines = BKE_gpencil_layer_addnew(gpd, "Lines", true, false);

  /* frames */
  /* NOTE: No need to check for existing, as this will take care of it for us */
  bGPDframe *frameFills = BKE_gpencil_frame_addnew(Fills, scene->r.cfra);
  bGPDframe *frameLines = BKE_gpencil_frame_addnew(Lines, scene->r.cfra);

  /* generate strokes */
  gps = BKE_gpencil_stroke_add(frameFills, color_Skin, 270, 75, false);
  ED_gpencil_stroke_init_data(gps, data0, 270, mat);
  BKE_gpencil_stroke_geometry_update(gpd, gps);

  gps = BKE_gpencil_stroke_add(frameFills, color_Skin_Shadow, 33, 60, false);
  ED_gpencil_stroke_init_data(gps, data1, 33, mat);
  BKE_gpencil_stroke_geometry_update(gpd, gps);

  gps = BKE_gpencil_stroke_add(frameFills, color_Skin_Shadow, 18, 60, false);
  ED_gpencil_stroke_init_data(gps, data2, 18, mat);
  BKE_gpencil_stroke_geometry_update(gpd, gps);

  gps = BKE_gpencil_stroke_add(frameFills, color_Skin_Light, 64, 60, false);
  ED_gpencil_stroke_init_data(gps, data3, 64, mat);
  BKE_gpencil_stroke_geometry_update(gpd, gps);

  gps = BKE_gpencil_stroke_add(frameFills, color_Skin_Light, 33, 60, false);
  ED_gpencil_stroke_init_data(gps, data4, 33, mat);
  BKE_gpencil_stroke_geometry_update(gpd, gps);

  gps = BKE_gpencil_stroke_add(frameFills, color_Skin_Light, 64, 60, false);
  ED_gpencil_stroke_init_data(gps, data5, 64, mat);
  BKE_gpencil_stroke_geometry_update(gpd, gps);

  gps = BKE_gpencil_stroke_add(frameFills, color_Skin_Light, 33, 60, false);
  ED_gpencil_stroke_init_data(gps, data6, 33, mat);
  BKE_gpencil_stroke_geometry_update(gpd, gps);

  gps = BKE_gpencil_stroke_add(frameFills, color_Skin_Light, 18, 40, false);
  ED_gpencil_stroke_init_data(gps, data7, 18, mat);
  BKE_gpencil_stroke_geometry_update(gpd, gps);

  gps = BKE_gpencil_stroke_add(frameFills, color_Eyes, 49, 60, false);
  ED_gpencil_stroke_init_data(gps, data8, 49, mat);
  BKE_gpencil_stroke_geometry_update(gpd, gps);

  gps = BKE_gpencil_stroke_add(frameFills, color_Skin_Shadow, 33, 60, false);
  ED_gpencil_stroke_init_data(gps, data9, 33, mat);
  BKE_gpencil_stroke_geometry_update(gpd, gps);

  gps = BKE_gpencil_stroke_add(frameFills, color_Eyes, 49, 60, false);
  ED_gpencil_stroke_init_data(gps, data10, 49, mat);
  BKE_gpencil_stroke_geometry_update(gpd, gps);

  gps = BKE_gpencil_stroke_add(frameFills, color_Skin_Shadow, 18, 40, false);
  ED_gpencil_stroke_init_data(gps, data11, 18, mat);
  BKE_gpencil_stroke_geometry_update(gpd, gps);

  gps = BKE_gpencil_stroke_add(frameFills, color_Skin_Shadow, 18, 40, false);
  ED_gpencil_stroke_init_data(gps, data12, 18, mat);
  BKE_gpencil_stroke_geometry_update(gpd, gps);

  gps = BKE_gpencil_stroke_add(frameLines, color_Black, 33, 60, false);
  ED_gpencil_stroke_init_data(gps, data13, 33, mat);
  BKE_gpencil_stroke_geometry_update(gpd, gps);

  gps = BKE_gpencil_stroke_add(frameLines, color_Black, 33, 60, false);
  ED_gpencil_stroke_init_data(gps, data14, 33, mat);
  BKE_gpencil_stroke_geometry_update(gpd, gps);

  gps = BKE_gpencil_stroke_add(frameLines, color_Black, 65, 60, false);
  ED_gpencil_stroke_init_data(gps, data15, 65, mat);
  BKE_gpencil_stroke_geometry_update(gpd, gps);

  gps = BKE_gpencil_stroke_add(frameLines, color_Black, 34, 60, false);
  ED_gpencil_stroke_init_data(gps, data16, 34, mat);
  BKE_gpencil_stroke_geometry_update(gpd, gps);

  gps = BKE_gpencil_stroke_add(frameLines, color_Black, 33, 60, false);
  ED_gpencil_stroke_init_data(gps, data17, 33, mat);
  BKE_gpencil_stroke_geometry_update(gpd, gps);

  gps = BKE_gpencil_stroke_add(frameLines, color_Black, 33, 40, false);
  ED_gpencil_stroke_init_data(gps, data18, 33, mat);
  BKE_gpencil_stroke_geometry_update(gpd, gps);

  gps = BKE_gpencil_stroke_add(frameLines, color_Black, 34, 40, false);
  ED_gpencil_stroke_init_data(gps, data19, 34, mat);
  BKE_gpencil_stroke_geometry_update(gpd, gps);

  gps = BKE_gpencil_stroke_add(frameLines, color_Black, 33, 60, false);
  ED_gpencil_stroke_init_data(gps, data20, 33, mat);
  BKE_gpencil_stroke_geometry_update(gpd, gps);

  gps = BKE_gpencil_stroke_add(frameLines, color_Black, 64, 60, false);
  ED_gpencil_stroke_init_data(gps, data21, 64, mat);
  BKE_gpencil_stroke_geometry_update(gpd, gps);

  gps = BKE_gpencil_stroke_add(frameLines, color_Pupils, 26, 60, false);
  ED_gpencil_stroke_init_data(gps, data22, 26, mat);
  BKE_gpencil_stroke_geometry_update(gpd, gps);

  gps = BKE_gpencil_stroke_add(frameLines, color_Pupils, 26, 60, false);
  ED_gpencil_stroke_init_data(gps, data23, 26, mat);
  BKE_gpencil_stroke_geometry_update(gpd, gps);

  gps = BKE_gpencil_stroke_add(frameLines, color_Black, 33, 60, false);
  ED_gpencil_stroke_init_data(gps, data24, 33, mat);
  BKE_gpencil_stroke_geometry_update(gpd, gps);

  gps = BKE_gpencil_stroke_add(frameLines, color_Black, 18, 40, false);
  ED_gpencil_stroke_init_data(gps, data25, 18, mat);
  BKE_gpencil_stroke_geometry_update(gpd, gps);

  gps = BKE_gpencil_stroke_add(frameLines, color_Black, 18, 40, false);
  ED_gpencil_stroke_init_data(gps, data26, 18, mat);
  BKE_gpencil_stroke_geometry_update(gpd, gps);

  gps = BKE_gpencil_stroke_add(frameLines, color_Black, 33, 60, false);
  ED_gpencil_stroke_init_data(gps, data27, 33, mat);
  BKE_gpencil_stroke_geometry_update(gpd, gps);

  /* update depsgraph */
  DEG_id_tag_update(&gpd->id, ID_RECALC_TRANSFORM | ID_RECALC_GEOMETRY);
  gpd->flag |= GP_DATA_CACHE_IS_DIRTY;
}

/* ***************************************************************** */
