; AleoBFT Library
;
; Copyright (C) 2024 Provable Inc.
;
; License: See the LICENSE file distributed with this library.
;
; Authors: Alessandro Coglio (www.alessandrocoglio.info)
;          Eric McCarthy (bendyarm on GitHub)

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

(in-package "ALEOBFT-STATIC")

(include-book "initialization")
(include-book "transitions")

(local (include-book "kestrel/built-ins/disable" :dir :system))
(local (acl2::disable-most-builtin-logic-defuns))
(local (acl2::disable-builtin-rewrite-rules-for-defaults))
(set-induction-depth-limit 0)

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

(defxdoc+ invariant-no-self-endorsed
  :parents (correctness)
  :short "Invariant that the recorded author-round pairs
          of endorsed certificates are from other validators."
  :long
  (xdoc::topstring
   (xdoc::p
    "When a new certificate is created,
     it is endorsed by validators other than the author of the certificate:
     @(tsee create-certificate-possiblep) checks that condition.
     A @('create-certificate') event is the only one
     that extends the @('endorsed') component of a validator state.
     The other events either leave it unchanged
     or remove pairs from it.
     Thus, it follows that all the pairs have authors
     distinct from the validator."))
  :order-subtopics t
  :default-parent t)

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

(define address+pos-not-address-p ((pair address+posp)
                                   (addr addressp))
  :returns (yes/no booleanp)
  :short "Check that the address of an address-integer pair
          differs from a given address."
  :long
  (xdoc::topstring
   (xdoc::p
    "The pair represents an author-round pair in a validator,
     and the address represents the validator's address."))
  (not (equal (address+pos->address pair) addr)))

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

(define address+pos-set-not-address-p ((pairs address+pos-setp)
                                       (addr addressp))
  :returns (yes/no booleanp)
  :short "Check that all the addresses in a set of address-integer pairs
          differ from a given address."
  :long
  (xdoc::topstring
   (xdoc::p
    "The set represents the set of recorded author-round pairs in a validator,
     and the address represents the validator's address."))
  (or (set::emptyp pairs)
      (and (address+pos-not-address-p (set::head pairs) addr)
           (address+pos-set-not-address-p (set::tail pairs) addr)))

  ///

  (defruled address+pos-set-not-address-p-element
    (implies (and (address+pos-set-not-address-p pairs addr)
                  (set::in pair pairs))
             (address+pos-not-address-p pair addr))
    :induct t)

  (defruled address+pos-set-not-address-p-subset
    (implies (and (address+pos-set-not-address-p pairs addr)
                  (set::subset pairs0 pairs))
             (address+pos-set-not-address-p pairs0 addr))
    :induct t
    :enable (set::subset
             address+pos-set-not-address-p
             address+pos-set-not-address-p-element))

  (defruled address+pos-set-not-address-p-of-insert
    (equal (address+pos-set-not-address-p (set::insert pair pairs) addr)
           (and (address+pos-not-address-p pair addr)
                (address+pos-set-not-address-p pairs addr)))
    :induct (set::weak-insert-induction pair pairs)
    :enable address+pos-set-not-address-p-element)

  (defruled address+pos-set-not-address-p-of-delete
    (implies (address+pos-set-not-address-p pairs addr)
             (address+pos-set-not-address-p (set::delete pair pairs)
                                            addr))
    :enable address+pos-set-not-address-p-subset))

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

(define validator-endorsed-not-self-p ((val addressp)
                                       (vstate validator-statep))
  :returns (yes/no booleanp)
  :short "Definition of the invariant at the validator level:
          the author-round pairs do not include the validator's address."
  (address+pos-set-not-address-p
   (validator-state->endorsed vstate) val))

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

(define validators-endorsed-not-self-p ((vstates validators-statep))
  :returns (yes/no booleanp)
  :short "Definition of the invariant at the validator map level:
          every correct validator in the map
          satisfies the validator-level invariant."
  :long
  (xdoc::topstring
   (xdoc::p
    "This lifts the invariant from single validators to validator maps.
     There is no requirement on faulty validators.")
   (xdoc::p
    "This definition and accompanying theorem
     could be automatically generated by a suitable macro."))
  (b* (((when (omap::emptyp vstates)) t)
       ((mv val vstate) (omap::head vstates))
       ((unless (or (not vstate)
                    (validator-endorsed-not-self-p val vstate)))
        nil))
    (validators-endorsed-not-self-p (omap::tail vstates)))
  ///

  (defruled validator-endorsed-not-self-p-of-lookup
    (implies (and (validators-endorsed-not-self-p vstates)
                  (omap::assoc val vstates)
                  (omap::lookup val vstates))
             (validator-endorsed-not-self-p val (omap::lookup val vstates)))
    :induct t
    :enable omap::lookup)

  (defruled validators-endorsed-not-self-p-of-update
    (implies (and (validators-endorsed-not-self-p vstates)
                  (validator-endorsed-not-self-p val vstate)
                  (addressp val))
             (validators-endorsed-not-self-p
              (omap::update val vstate vstates)))
    :induct t
    :enable (omap::update
             omap::head
             omap::tail)))

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

(define system-endorsed-not-self-p ((systate system-statep))
  :returns (yes/no booleanp)
  :short "Definition of the invariant at the system level:
          every correct validator in the system
          satisfies the validator-level invariant."
  :long
  (xdoc::topstring
   (xdoc::p
    "This lifts the invariant from the validator map to the system state.")
   (xdoc::p
    "This definition and accompanying theorems
     could be automatically generated by a suitable macro."))
  (validators-endorsed-not-self-p (system-state->validators systate))
  :hooks (:fix)
  ///

  (defruled system-endorsed-not-self-p-of-update-network-state
    (equal (system-endorsed-not-self-p (update-network-state network systate))
           (system-endorsed-not-self-p systate))
    :enable update-network-state)

  (defruled system-endorsed-not-self-p-of-update-validator-state
    (implies (and (system-endorsed-not-self-p systate)
                  (validator-endorsed-not-self-p val vstate)
                  (addressp val)
                  (validator-statep vstate))
             (system-endorsed-not-self-p
              (update-validator-state val vstate systate)))
    :enable update-validator-state)

  (defruled validator-endorsed-not-self-p-of-get-validator-state
    (implies (and (system-endorsed-not-self-p systate)
                  (set::in val (correct-addresses systate)))
             (validator-endorsed-not-self-p val
                                            (get-validator-state val systate)))
    :enable (get-validator-state
             correct-addresses
             in-of-correct-addresses-loop
             lookup-nonnil-of-correct-addresses
             validator-endorsed-not-self-p-of-lookup)))

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

(defruled validator-endorsed-not-self-p-of-validator-init
  :short "Establishment of the invariant at the validator level:
          the invariant holds on the initial validator state."
  :long
  (xdoc::topstring
   (xdoc::p
    "The initial set of author-round pairs is empty, so the proof is trivial."))
  (validator-endorsed-not-self-p val (validator-init))
  :enable (validator-endorsed-not-self-p
           validator-init
           address+pos-set-not-address-p))

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

(defruled validators-endorsed-not-self-p-when-validators-state-initp
  :short "Establishment of the invariant at the validator map level:
          the invariant holds on any initial validator map."
  :long
  (xdoc::topstring
   (xdoc::p
    "This just follows structurally from
     the theorem on a single validator."))
  (implies (validators-state-initp vstates)
           (validators-endorsed-not-self-p vstates))
  :induct t
  :enable (validators-endorsed-not-self-p
           validators-state-initp
           validator-endorsed-not-self-p-of-validator-init))

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

(defruled system-endorsed-not-self-p-when-system-state-initp
  :short "Establishment of the invariant at the system level:
          the invariant holds on any initial system state."
  :long
  (xdoc::topstring
   (xdoc::p
    "This just follows from the theorem on the validator map.
     The network is not involved in the invariant."))
  (implies (system-state-initp systate)
           (system-endorsed-not-self-p systate))
  :enable (system-state-initp
           system-endorsed-not-self-p))

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

(defsection system-endorsed-not-self-p-of-add-endorsed
  :short "Preservation of the invariant by
          the addition of endorsed author-round pairs."
  :long
  (xdoc::topstring
   (xdoc::p
    "A @('create-certificate') event makes use of
     @(tsee add-endorsed).
     If the author is distinct from the endorser for the new added pair,
     and if the invariant holds for the old pairs,
     the invariant holds for the new pairs.
     The distinctness hypotheses are established
     in the proof of the theorem for @('create-certificate') events later."))

  (defruled validator-endorsed-not-self-p-of-add-endorsed-val
    (implies (and (validator-endorsed-not-self-p val vstate)
                  (not (equal author val))
                  (addressp author))
             (validator-endorsed-not-self-p
              val (add-endorsed-val author round vstate)))
    :enable (add-endorsed-val
             validator-endorsed-not-self-p
             address+pos-not-address-p
             address+pos-set-not-address-p-of-insert
             set::expensive-rules))

  (defruled system-endorsed-not-self-p-of-add-endorsed
    (implies (and (system-endorsed-not-self-p systate)
                  (not (set::in author endorsers))
                  (addressp author)
                  (address-setp endorsers)
                  (set::subset endorsers (all-addresses systate)))
             (system-endorsed-not-self-p
              (add-endorsed endorsers author round systate)))
    :induct t
    :enable (in-correct-validator-addresess-when-get-validator-state
             add-endorsed
             address+pos-not-address-p
             system-endorsed-not-self-p-of-update-validator-state
             validator-endorsed-not-self-p-of-get-validator-state
             validator-endorsed-not-self-p-of-add-endorsed-val
             set::expensive-rules)
    :prep-lemmas
    ((defrule lemma
       (implies (and (not (set::emptyp set))
                     (not (set::in elem set)))
                (not (equal elem (set::head set))))))))

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

(defsection system-endorsed-not-self-p-of-create-certificate-next
  :short "Preservation of the invariant by @('create-certificate') events."
  :long
  (xdoc::topstring
   (xdoc::p
    "This is one of the only two kinds of events that modify author-round pairs.
     The @(tsee create-certificate-possiblep) ensures that
     the author is distinct from the endorsers,
     thus establishing the hypothesis for
     the theorem about @(tsee add-endorsed)."))

  (defruled validator-endorsed-not-self-p-of-create-certificate-next-val
    (implies (validator-endorsed-not-self-p val vstate)
             (validator-endorsed-not-self-p
              val (create-certificate-next-val cert vstate)))
    :enable (validator-endorsed-not-self-p
             create-certificate-next-val))

  (defruled system-endorsed-not-self-p-of-create-certificate-next
    (implies (and (system-endorsed-not-self-p systate)
                  (create-certificate-possiblep cert systate))
             (system-endorsed-not-self-p
              (create-certificate-next cert systate)))
    :enable (create-certificate-possiblep
             create-certificate-next
             system-endorsed-not-self-p-of-update-network-state
             system-endorsed-not-self-p-of-update-validator-state
             validator-endorsed-not-self-p-of-get-validator-state
             system-endorsed-not-self-p-of-add-endorsed
             validator-endorsed-not-self-p-of-create-certificate-next-val)))

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

(defsection system-endorsed-not-self-p-of-receive-certificate-next
  :short "Preservation of the invariant by @('receive-certificate') events."
  :long
  (xdoc::topstring
   (xdoc::p
    "This is one of the only two kinds of events that modify author-round pairs,
     but it removes a pair from a set,
     so the proof is easy."))

  (defruled validator-endorsed-not-self-p-of-receive-certificate-next-val
    (implies (validator-endorsed-not-self-p val vstate)
             (validator-endorsed-not-self-p
              val (receive-certificate-next-val cert vstate)))
    :enable (validator-endorsed-not-self-p
             receive-certificate-next-val
             address+pos-set-not-address-p-of-delete))

  (defruled system-endorsed-not-self-p-of-receive-certificate-next
    (implies (and (system-endorsed-not-self-p systate)
                  (receive-certificate-possiblep msg systate))
             (system-endorsed-not-self-p
              (receive-certificate-next msg systate)))
    :enable (receive-certificate-possiblep
             receive-certificate-next
             system-endorsed-not-self-p-of-update-network-state
             system-endorsed-not-self-p-of-update-validator-state
             validator-endorsed-not-self-p-of-get-validator-state
             validator-endorsed-not-self-p-of-receive-certificate-next-val)))

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

(defsection system-endorsed-not-self-p-of-store-certificate-next
  :short "Preservation of the invariant by @('store-certificate') events."
  :long
  (xdoc::topstring
   (xdoc::p
    "This is easy to prove,
     because that kind of events does not modify author-round pairs."))

  (defruled validator-endorsed-not-self-p-of-store-certificate-next-val
    (implies (validator-endorsed-not-self-p val vstate)
             (validator-endorsed-not-self-p
              val (store-certificate-next-val cert vstate)))
    :enable (validator-endorsed-not-self-p
             store-certificate-next-val))

  (defruled system-endorsed-not-self-p-of-store-certificate-next
    (implies (and (system-endorsed-not-self-p systate)
                  (store-certificate-possiblep cert val systate))
             (system-endorsed-not-self-p
              (store-certificate-next cert val systate)))
    :enable (store-certificate-possiblep
             store-certificate-next
             system-endorsed-not-self-p-of-update-validator-state
             validator-endorsed-not-self-p-of-get-validator-state
             validator-endorsed-not-self-p-of-store-certificate-next-val)))

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

(defsection system-endorsed-not-self-p-of-advance-round-next
  :short "Preservation of the invariant by @('advance-round') events."
  :long
  (xdoc::topstring
   (xdoc::p
    "This is easy to prove,
     because that kind of events does not modify author-round pairs."))

  (defruled validator-endorsed-not-self-p-of-advance-round-next-val
    (implies (validator-endorsed-not-self-p val vstate)
             (validator-endorsed-not-self-p
              val (advance-round-next-val round vstate)))
    :enable (validator-endorsed-not-self-p
             advance-round-next-val))

  (defruled system-endorsed-not-self-p-of-advance-round-next
    (implies (and (system-endorsed-not-self-p systate)
                  (advance-round-possiblep val systate))
             (system-endorsed-not-self-p
              (advance-round-next val systate)))
    :enable (advance-round-possiblep
             advance-round-next
             system-endorsed-not-self-p-of-update-validator-state
             validator-endorsed-not-self-p-of-get-validator-state
             validator-endorsed-not-self-p-of-advance-round-next-val)))

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

(defsection system-endorsed-not-self-p-of-commit-anchors-next
  :short "Preservation of the invariant by @('commit-anchors') events."
  :long
  (xdoc::topstring
   (xdoc::p
    "This is easy to prove,
     because that kind of events does not modify author-round pairs."))

  (defruled validator-endorsed-not-self-p-of-commit-anchors-next-val
    (implies (validator-endorsed-not-self-p val vstate)
             (validator-endorsed-not-self-p
              val (commit-anchors-next-val vals vstate)))
    :enable (validator-endorsed-not-self-p
             commit-anchors-next-val))

  (defruled system-endorsed-not-self-p-of-commit-anchors-next
    (implies (and (system-endorsed-not-self-p systate)
                  (commit-anchors-possiblep val systate))
             (system-endorsed-not-self-p
              (commit-anchors-next val systate)))
    :enable (commit-anchors-possiblep
             commit-anchors-next
             system-endorsed-not-self-p-of-update-validator-state
             validator-endorsed-not-self-p-of-get-validator-state
             validator-endorsed-not-self-p-of-commit-anchors-next-val)))

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

(defsection system-endorsed-not-self-p-of-timer-expires-next
  :short "Preservation of the invariant by @('timer-expires') events."
  :long
  (xdoc::topstring
   (xdoc::p
    "This is easy to prove,
     because that kind of events does not modify author-round pairs."))

  (defruled validator-endorsed-not-self-p-of-timer-expires-next-val
    (implies (validator-endorsed-not-self-p val vstate)
             (validator-endorsed-not-self-p
              val (timer-expires-next-val vstate)))
    :enable (validator-endorsed-not-self-p
             timer-expires-next-val))

  (defruled system-endorsed-not-self-p-of-timer-expires-next
    (implies (and (system-endorsed-not-self-p systate)
                  (timer-expires-possiblep val systate))
             (system-endorsed-not-self-p (timer-expires-next val systate)))
    :enable (timer-expires-possiblep
             timer-expires-next
             system-endorsed-not-self-p-of-update-validator-state
             validator-endorsed-not-self-p-of-get-validator-state
             validator-endorsed-not-self-p-of-timer-expires-next-val)))

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

(defruled system-endorsed-not-self-p-of-event-next
  :short "Preservation of the invariant by all events."
  :long
  (xdoc::topstring
   (xdoc::p
    "This follows from the theorems about the various kinds of events."))
  (implies (and (system-endorsed-not-self-p systate)
                (event-possiblep event systate))
           (system-endorsed-not-self-p
            (event-next event systate)))
  :enable (event-possiblep event-next))

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

(defruled system-endorsed-not-self-p-of-events-next
  :short "Preservation of the invariant by every sequence of events."
  (implies (and (system-statep systate)
                (system-endorsed-not-self-p systate)
                (events-possiblep events systate))
           (system-endorsed-not-self-p (events-next events systate)))
  :induct t
  :disable ((:e tau-system))
  :enable (events-next
           events-possiblep
           system-endorsed-not-self-p-of-event-next))

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

(defruled system-endorsed-not-self-p-when-reachable
  :short "The invariant holds in every reachable state."
  :long
  (xdoc::topstring
   (xdoc::p
    "Reachable states are characterized by an initial state and
     a sequence of possible events from that initial state."))
  (implies (and (system-statep systate)
                (system-state-initp systate)
                (events-possiblep events systate))
           (system-endorsed-not-self-p (events-next events systate)))
  :disable ((:e tau-system))
  :enable (system-endorsed-not-self-p-when-system-state-initp
           system-endorsed-not-self-p-of-events-next))
